<?php
/**
 * Sync Manager för Bokför Ehandel
 * 
 * Hanterar all synkroniseringslogik för ordrar, betalningar och utbetalningar
 * 
 * @package Bokfor_Ehandel
 * @since 1.2.0
 */

// Om denna fil anropas direkt, avbryt.
if (!defined('ABSPATH')) {
    die;
}

/**
 * Sync Manager klass
 */
class Bokfor_Ehandel_Sync_Manager {
    
    /**
     * API Client-instans
     * 
     * @var Bokfor_Ehandel_API_Client
     */
    private $api_client;
    
    /**
     * Settings-instans
     * 
     * @var Bokfor_Ehandel_Settings
     */
    private $settings;
    
    /**
     * Logger-instans
     * 
     * @var Bokfor_Ehandel_Logger
     */
    private $logger;
    
    /**
     * Batch-storlek för synkroniseringar
     * 
     * @var int
     */
    private $batch_size = 50;
    
    /**
     * Konstruktor
     * 
     * @param Bokfor_Ehandel_API_Client $api_client API Client-instans
     * @param Bokfor_Ehandel_Settings $settings Settings-instans
     * @param Bokfor_Ehandel_Logger $logger Logger-instans
     */
    public function __construct($api_client, $settings, $logger) {
        $this->api_client = $api_client;
        $this->settings = $settings;
        $this->logger = $logger;
    }
    
    /**
     * Synkronisera ordrar
     * 
     * @param string|null $start_date Startdatum (YYYY-MM-DD)
     * @param string|null $end_date Slutdatum (YYYY-MM-DD)
     * @param string $operation Operationstyp ('orders', 'payments', 'all')
     * @return array Resultat från synkroniseringen
     */
    public function sync_orders($start_date = null, $end_date = null, $operation = 'orders') {
        $this->logger->info("Startar synkronisering av ordrar. Operation: {$operation}, Start: {$start_date}, Slut: {$end_date}");
        
        // Förbered data för API-anrop
        $api_data = $this->prepare_sync_data($operation, $start_date, $end_date);
        
        // Gör API-anrop
        $response = $this->api_client->post('/api/integration/run', $api_data);
        
        if (is_wp_error($response) || (isset($response['status_code']) && $response['status_code'] >= 400)) {
            $error_msg = is_wp_error($response) ? $response->get_error_message() : ($response['error'] ?? 'Okänt fel');
            $this->logger->error("Synkronisering misslyckades: {$error_msg}");
            return array(
                'success' => false,
                'message' => $error_msg
            );
        }
        
        $this->logger->info("Synkronisering lyckades");
        return array(
            'success' => true,
            'data' => $response['data'] ?? null
        );
    }
    
    /**
     * Synkronisera betalningar
     * 
     * @param string|null $start_date Startdatum (YYYY-MM-DD)
     * @param string|null $end_date Slutdatum (YYYY-MM-DD)
     * @return array Resultat från synkroniseringen
     */
    public function sync_payments($start_date = null, $end_date = null) {
        return $this->sync_orders($start_date, $end_date, 'payments');
    }
    
    /**
     * Synkronisera Klarna-utbetalningar
     * 
     * @return array Resultat från synkroniseringen
     */
    public function sync_klarna_settlements() {
        $this->logger->info("Startar synkronisering av Klarna-utbetalningar");
        
        // Kontrollera om Klarna är aktiverat
        if (!$this->settings->get('klarna_enabled', false)) {
            return array(
                'success' => false,
                'message' => 'Klarna är inte aktiverat'
            );
        }
        
        // Förbered data för API-anrop
        $api_data = $this->prepare_sync_data('klarna_settlements');
        
        // Gör API-anrop
        $response = $this->api_client->post('/api/integration/run', $api_data);
        
        if (is_wp_error($response) || (isset($response['status_code']) && $response['status_code'] >= 400)) {
            $error_msg = is_wp_error($response) ? $response->get_error_message() : ($response['error'] ?? 'Okänt fel');
            $this->logger->error("Klarna-synkronisering misslyckades: {$error_msg}");
            return array(
                'success' => false,
                'message' => $error_msg
            );
        }
        
        $this->logger->info("Klarna-synkronisering lyckades");
        return array(
            'success' => true,
            'data' => $response['data'] ?? null
        );
    }
    
    /**
     * Synkronisera Stripe-utbetalningar
     * 
     * @return array Resultat från synkroniseringen
     */
    public function sync_stripe_settlements() {
        $this->logger->info("Startar synkronisering av Stripe-utbetalningar");
        
        // Kontrollera om Stripe är aktiverat
        if (!$this->settings->get('stripe_enabled', false)) {
            return array(
                'success' => false,
                'message' => 'Stripe är inte aktiverat'
            );
        }
        
        // Förbered data för API-anrop
        $api_data = $this->prepare_sync_data('stripe_settlements');
        
        // Gör API-anrop
        $response = $this->api_client->post('/api/integration/run', $api_data);
        
        if (is_wp_error($response) || (isset($response['status_code']) && $response['status_code'] >= 400)) {
            $error_msg = is_wp_error($response) ? $response->get_error_message() : ($response['error'] ?? 'Okänt fel');
            $this->logger->error("Stripe-synkronisering misslyckades: {$error_msg}");
            return array(
                'success' => false,
                'message' => $error_msg
            );
        }
        
        $this->logger->info("Stripe-synkronisering lyckades");
        return array(
            'success' => true,
            'data' => $response['data'] ?? null
        );
    }
    
    /**
     * Förbered synkroniseringsdata för API-anrop
     * 
     * @param string $operation Operationstyp
     * @param string|null $start_date Startdatum
     * @param string|null $end_date Slutdatum
     * @return array Förberedd data
     */
    private function prepare_sync_data($operation, $start_date = null, $end_date = null) {
        $accounts = $this->settings->get_accounts();
        
        $data = array(
            'operation' => $operation,
            'store_name' => $this->settings->get_store_name(),
            'company_name' => $this->settings->get_company_name(),
            'organization_number' => $this->settings->get_organization_number(),
            'upload_receipts' => $this->settings->get('upload_receipts', true),
            'upload_settlements' => $this->settings->get('upload_settlements', true),
            'journal_id' => $accounts['journal_id'],
            'klarna_journal_id' => $accounts['klarna_journal_id'],
            'stripe_journal_id' => $accounts['stripe_journal_id'],
            'accounts' => array(
                'bank_account' => $accounts['bank_account'],
                'receivables_account' => $accounts['receivables_account'],
                'fee_account' => $accounts['fee_account'],
                'input_vat_account' => $accounts['input_vat_account'],
                'sales_account' => $accounts['sales_account'],
                'vat_account' => $accounts['vat_account']
            )
        );
        
        // Lägg till betalningskonton
        $payment_accounts = $this->settings->get('payment_accounts', array());
        if (!empty($payment_accounts)) {
            $data['payment_accounts'] = $payment_accounts;
        }
        
        // Lägg till WooCommerce API-information
        $wc_api_url = $this->settings->get('wc_api_url', '');
        if (!empty($wc_api_url)) {
            $data['woocommerce'] = array(
                'api_url' => $wc_api_url,
                'consumer_key' => $this->settings->get('wc_consumer_key', ''),
                'consumer_secret' => $this->settings->get('wc_consumer_secret', '')
            );
        }
        
        // Lägg till datum om angivna
        if (!empty($start_date)) {
            $data['start_date'] = $start_date;
        }
        if (!empty($end_date)) {
            $data['end_date'] = $end_date;
        }
        
        // Lägg till Klarna-inställningar om aktiverat
        if ($this->settings->get('klarna_enabled', false)) {
            $data['klarna'] = array(
                'enabled' => true,
                'api_username' => $this->settings->get('klarna_api_username', ''),
                'api_password' => $this->settings->get('klarna_api_password', '')
            );
        }
        
        // Lägg till Stripe-inställningar om aktiverat
        if ($this->settings->get('stripe_enabled', false)) {
            $data['stripe'] = array(
                'enabled' => true,
                'api_key' => $this->settings->get('stripe_api_key', '')
            );
        }
        
        return $data;
    }
    
    /**
     * Synkronisera ordrar i batch
     * 
     * @param array $order_ids Array med order-ID:n
     * @param int $batch_size Storlek på varje batch
     * @return array Resultat från synkroniseringen
     */
    public function sync_orders_batch($order_ids, $batch_size = null) {
        if ($batch_size === null) {
            $batch_size = $this->batch_size;
        }
        
        $batches = array_chunk($order_ids, $batch_size);
        $results = array(
            'success' => 0,
            'failed' => 0,
            'total' => count($order_ids)
        );
        
        foreach ($batches as $batch) {
            // Synkronisera batch
            // TODO: Implementera batch-synkronisering när API stödjer det
            $this->logger->info("Synkroniserar batch med " . count($batch) . " ordrar");
        }
        
        return $results;
    }
}

