<?php
/**
 * Settings Manager för Bokför Ehandel
 * 
 * Centraliserad hantering av plugin-inställningar
 * 
 * @package Bokfor_Ehandel
 * @since 1.2.0
 */

// Om denna fil anropas direkt, avbryt.
if (!defined('ABSPATH')) {
    die;
}

/**
 * Settings Manager klass
 */
class Bokfor_Ehandel_Settings {
    
    /**
     * Option prefix
     * 
     * @var string
     */
    private $prefix = 'bokforehandel_';
    
    /**
     * Cache för inställningar
     * 
     * @var array
     */
    private $cache = array();
    
    /**
     * Hämta inställning
     * 
     * @param string $key Inställningsnyckel (utan prefix)
     * @param mixed $default Standardvärde om inställningen saknas
     * @return mixed Inställningsvärde
     */
    public function get($key, $default = '') {
        $option_name = $this->prefix . $key;
        
        // Kontrollera cache först
        if (isset($this->cache[$option_name])) {
            return $this->cache[$option_name];
        }
        
        // Hämta från databas
        $value = get_option($option_name, $default);
        
        // Spara i cache
        $this->cache[$option_name] = $value;
        
        return $value;
    }
    
    /**
     * Spara inställning
     * 
     * @param string $key Inställningsnyckel (utan prefix)
     * @param mixed $value Värde att spara
     * @return bool Om sparningen lyckades
     */
    public function set($key, $value) {
        $option_name = $this->prefix . $key;
        
        // Spara i databas
        $result = update_option($option_name, $value);
        
        // Uppdatera cache
        if ($result) {
            $this->cache[$option_name] = $value;
        }
        
        return $result;
    }
    
    /**
     * Validera inställning
     * 
     * @param string $key Inställningsnyckel
     * @param mixed $value Värde att validera
     * @return bool|string True om giltigt, felmeddelande om ogiltigt
     */
    public function validate($key, $value) {
        // Specifika valideringar för olika fält
        switch ($key) {
            case 'organization_number':
                // Validera organisationsnummer (använd befintlig valideringsmetod)
                // Detta kommer att anropas från huvudklassen
                return true; // Validering görs i huvudklassen
                
            case 'sales_account':
            case 'vat_account':
            case 'bank_account':
            case 'receivables_account':
            case 'fee_account':
            case 'input_vat_account':
                // Validera kontonummer (4 siffror)
                return preg_match('/^\d{4}$/', $value) === 1;
                
            case 'billing_email':
                return is_email($value);
                
            case 'sync_enabled':
                return in_array($value, array('yes', 'no', '1', '0', true, false));
                
            default:
                return true; // Ingen specifik validering
        }
    }
    
    /**
     * Hämta alla inställningar
     * 
     * @return array Alla inställningar
     */
    public function get_all() {
        global $wpdb;
        
        $options = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT option_name, option_value FROM {$wpdb->options} WHERE option_name LIKE %s",
                $this->prefix . '%'
            ),
            ARRAY_A
        );
        
        $settings = array();
        foreach ($options as $option) {
            $key = str_replace($this->prefix, '', $option['option_name']);
            $settings[$key] = maybe_unserialize($option['option_value']);
        }
        
        return $settings;
    }
    
    /**
     * Type-safe getters för specifika inställningar
     */
    
    /**
     * Hämta företagsnamn
     * 
     * @return string
     */
    public function get_company_name() {
        return $this->get('company_name', get_bloginfo('name'));
    }
    
    /**
     * Hämta butiksnamn
     * 
     * @return string
     */
    public function get_store_name() {
        return $this->get('store_name', get_bloginfo('name'));
    }
    
    /**
     * Hämta organisationsnummer
     * 
     * @return string
     */
    public function get_organization_number() {
        return $this->get('organization_number', '');
    }
    
    /**
     * Hämta BL User Key
     * 
     * @return string
     */
    public function get_bl_user_key() {
        return $this->get('bl_user_key', '');
    }
    
    /**
     * Hämta User ID
     * 
     * @return string
     */
    public function get_user_id() {
        return $this->get('user_id', '');
    }
    
    /**
     * Hämta integrationsstatus
     * 
     * @return string
     */
    public function get_integration_status() {
        return $this->get('integration_status', 'off');
    }
    
    /**
     * Hämta synkroniseringsinställningar
     * 
     * @return array
     */
    public function get_sync_settings() {
        return array(
            'enabled' => $this->get('sync_enabled', 'yes'),
            'frequency' => $this->get('sync_frequency', 'daily'),
            'period' => $this->get('sync_period', 'lastday')
        );
    }
    
    /**
     * Hämta bokföringskonton
     * 
     * @return array
     */
    public function get_accounts() {
        return array(
            'sales_account' => $this->get('sales_account', BOKFOREHANDEL_DEFAULT_SALES_ACCOUNT),
            'vat_account' => $this->get('vat_account', BOKFOREHANDEL_DEFAULT_VAT_ACCOUNT),
            'bank_account' => $this->get('bank_account', BOKFOREHANDEL_DEFAULT_BANK_ACCOUNT),
            'receivables_account' => $this->get('receivables_account', BOKFOREHANDEL_DEFAULT_RECEIVABLES_ACCOUNT),
            'fee_account' => $this->get('fee_account', BOKFOREHANDEL_DEFAULT_FEE_ACCOUNT),
            'input_vat_account' => $this->get('input_vat_account', BOKFOREHANDEL_DEFAULT_INPUT_VAT_ACCOUNT),
            'journal_id' => $this->get('journal_id', 'W'),
            'klarna_journal_id' => $this->get('klarna_journal_id', 'K'),
            'stripe_journal_id' => $this->get('stripe_journal_id', 'S')
        );
    }
    
    /**
     * Hämta butikens valuta
     *
     * @return string
     */
    public function get_store_currency() {
        // Försök hämta från WooCommerce-inställningar
        if (function_exists('wc_get_base_currency')) {
            return wc_get_base_currency();
        }

        // Fallback till inställning
        return $this->get('store_currency', 'SEK');
    }

    /**
     * Spara butikens valuta
     *
     * @param string $currency Valutakod (ISO 4217)
     * @return bool
     */
    public function set_store_currency($currency) {
        return $this->set('store_currency', sanitize_text_field($currency));
    }

    /**
     * Hämta valutainformation med fallback
     *
     * @return array Valutainformation
     */
    public function get_currency_info() {
        $currency = $this->get_store_currency();

        $supported_currencies = array(
            'SEK' => array('name' => 'Svensk Krona', 'symbol' => 'kr', 'decimal_places' => 2),
            'EUR' => array('name' => 'Euro', 'symbol' => '€', 'decimal_places' => 2),
            'USD' => array('name' => 'US Dollar', 'symbol' => '$', 'decimal_places' => 2),
            'NOK' => array('name' => 'Norsk Krona', 'symbol' => 'kr', 'decimal_places' => 2),
            'DKK' => array('name' => 'Dansk Krona', 'symbol' => 'kr', 'decimal_places' => 2)
        );

        return isset($supported_currencies[$currency])
            ? $supported_currencies[$currency]
            : $supported_currencies['SEK'];
    }

    /**
     * Rensa cache
     *
     * @return void
     */
    public function clear_cache() {
        $this->cache = array();
    }
}

