<?php
/**
 * Admin Manager för Bokför Ehandel
 * 
 * Hanterar admin-funktionalitet: menyer, scripts, sidor
 * 
 * @package Bokfor_Ehandel
 * @since 1.2.0
 */

// Om denna fil anropas direkt, avbryt.
if (!defined('ABSPATH')) {
    die;
}

/**
 * Admin Manager klass
 */
class Bokfor_Ehandel_Admin {
    
    /**
     * Settings-instans
     * 
     * @var Bokfor_Ehandel_Settings
     */
    private $settings;
    
    /**
     * Logger-instans
     * 
     * @var Bokfor_Ehandel_Logger
     */
    private $logger;
    
    /**
     * Konstruktor
     * 
     * @param Bokfor_Ehandel_Settings $settings Settings-instans
     * @param Bokfor_Ehandel_Logger $logger Logger-instans
     */
    public function __construct($settings, $logger) {
        $this->settings = $settings;
        $this->logger = $logger;
    }
    
    /**
     * Initiera admin-funktionalitet
     * 
     * @return void
     */
    public function init() {
        // Registrera admin-menyer
        add_action('admin_menu', array($this, 'add_admin_menu'));
        
        // Ladda admin scripts och styles
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        
        // Hantera omdirigeringar
        add_action('current_screen', array($this, 'handle_admin_redirects'));
    }
    
    /**
     * Lägg till admin-menyer
     * 
     * @return void
     */
    public function add_admin_menu() {
        // Huvudmeny
        add_menu_page(
            'Bokför Ehandel',
            'Bokför Ehandel',
            'manage_options',
            'bokforehandel-company',
            array($this, 'render_company_settings_page'),
            'dashicons-money-alt',
            58
        );
        
        // Företagsinställningar
        add_submenu_page(
            'bokforehandel-company',
            'Företagsinställningar',
            'Företagsinställningar',
            'manage_options',
            'bokforehandel-company',
            array($this, 'render_company_settings_page')
        );
        
        // Bokföringsinställningar
        add_submenu_page(
            'bokforehandel-company',
            'Bokföringsinställningar',
            'Bokföringsinställningar',
            'manage_options',
            'bokforehandel-accounting',
            array($this, 'render_accounting_settings_page')
        );
        
        // Synkronisering/Körningshistorik
        add_submenu_page(
            'bokforehandel-company',
            'Synkronisering',
            'Synkronisering',
            'manage_options',
            'bokforehandel-runs',
            array($this, 'render_runs_page')
        );
        
        // Hjälpsida
        add_submenu_page(
            'bokforehandel-company',
            'Hjälp',
            'Hjälp',
            'manage_options',
            'bokforehandel-help',
            array($this, 'render_help_page')
        );
    }
    
    /**
     * Ladda admin scripts och styles
     * 
     * @param string $hook Current admin page hook
     * @return void
     */
    public function enqueue_admin_scripts($hook) {
        if (strpos($hook, 'bokforehandel') === false) {
            return;
        }
        
        // CSS
        wp_enqueue_style('bokforehandel-admin', BOKFOREHANDEL_PLUGIN_URL . 'assets/css/admin.css', array(), BOKFOREHANDEL_VERSION);
        
        // JavaScript
        wp_enqueue_script('jquery-ui-tabs');
        wp_enqueue_script('jquery-ui-datepicker');
        
        wp_enqueue_script(
            'bokforehandel-admin',
            BOKFOREHANDEL_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery', 'jquery-ui-tabs', 'jquery-ui-datepicker'),
            BOKFOREHANDEL_VERSION,
            true
        );
        
        // Localize script
        wp_localize_script('bokforehandel-admin', 'bokforehandel', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('bokforehandel_nonce'),
            'user_id' => $this->settings->get_user_id(),
            'api_url' => BOKFOREHANDEL_API_URL,
            'integration_status' => $this->settings->get_integration_status(),
            'checking_status' => __('Kontrollerar integrationsstatus...', 'bokfor-ehandel'),
            'starting_sync' => __('Startar synkronisering...', 'bokfor-ehandel')
        ));
    }
    
    /**
     * Hantera omdirigeringar från gamla sidor
     * 
     * @return void
     */
    public function handle_admin_redirects() {
        $screen = get_current_screen();
        
        if (!$screen) {
            return;
        }
        
        // Omdirigera från gamla översiktssidan till företagsinställningar
        if ($screen->id === 'toplevel_page_bokforehandel') {
            wp_redirect(admin_url('admin.php?page=bokforehandel-company'));
            exit;
        }
        
        // Omdirigera från synkroniseringssidan till körningshistorik
        if ($screen->id === 'bokfor-ehandel_page_bokforehandel-sync') {
            wp_redirect(admin_url('admin.php?page=bokforehandel-runs'));
            exit;
        }
    }
    
    /**
     * Rendera företagsinställningssidan
     * 
     * @return void
     */
    public function render_company_settings_page() {
        include BOKFOREHANDEL_PLUGIN_DIR . 'templates/company_settings.php';
    }
    
    /**
     * Rendera bokföringsinställningssidan
     * 
     * @return void
     */
    public function render_accounting_settings_page() {
        include BOKFOREHANDEL_PLUGIN_DIR . 'templates/accounting_settings.php';
    }
    
    /**
     * Rendera körningshistorik-sidan
     * 
     * @return void
     */
    public function render_runs_page() {
        include BOKFOREHANDEL_PLUGIN_DIR . 'templates/runs.php';
    }
    
    /**
     * Rendera hjälpsidan
     * 
     * @return void
     */
    public function render_help_page() {
        include BOKFOREHANDEL_PLUGIN_DIR . 'templates/help.php';
    }
}

