<?php
/**
 * Logger för Bokför Ehandel
 * 
 * Centraliserad loggning med stöd för WooCommerce logger och error_log
 * 
 * @package Bokfor_Ehandel
 * @since 1.2.0
 */

// Om denna fil anropas direkt, avbryt.
if (!defined('ABSPATH')) {
    die;
}

/**
 * Logger klass
 */
class Bokfor_Ehandel_Logger {
    
    /**
     * Logga meddelande
     * 
     * @param string $message Meddelande att logga
     * @param string $level Loggnivå ('info', 'warning', 'error')
     * @return void
     */
    public function log($message, $level = 'info') {
        // Logga endast om WP_DEBUG är aktiverat eller om det är ett fel
        if (!defined('WP_DEBUG') || !WP_DEBUG) {
            if ($level !== 'error') {
                return; // Logga endast fel i produktion
            }
        }
        
        // Försök använda WooCommerce logger om tillgänglig
        if (function_exists('wc_get_logger') && $level === 'error') {
            $logger = wc_get_logger();
            $logger->error($message, array('source' => 'bokforehandel'));
            return;
        }
        
        $prefix = '[Bokför Ehandel]';
        
        switch ($level) {
            case 'error':
                $prefix .= ' [ERROR]';
                break;
            case 'warning':
                $prefix .= ' [WARNING]';
                break;
            default:
                $prefix .= ' [INFO]';
                break;
        }
        
        error_log($prefix . ' ' . $message);
    }
    
    /**
     * Logga fel
     * 
     * @param string $message Felmeddelande
     * @return void
     */
    public function error($message) {
        $this->log($message, 'error');
    }
    
    /**
     * Logga varning
     * 
     * @param string $message Varningsmeddelande
     * @return void
     */
    public function warning($message) {
        $this->log($message, 'warning');
    }
    
    /**
     * Logga info
     * 
     * @param string $message Informationsmeddelande
     * @return void
     */
    public function info($message) {
        $this->log($message, 'info');
    }
}

