<?php
/**
 * API Client för Bokför Ehandel
 * 
 * Centraliserad hantering av alla API-anrop till externa tjänster
 * 
 * @package Bokfor_Ehandel
 * @since 1.2.0
 */

// Om denna fil anropas direkt, avbryt.
if (!defined('ABSPATH')) {
    die;
}

/**
 * API Client klass
 */
class Bokfor_Ehandel_API_Client {
    
    /**
     * API base URL
     * 
     * @var string
     */
    private $api_url;
    
    /**
     * API Key för autentisering
     * 
     * @var string
     */
    private $api_key;
    
    /**
     * User ID för autentisering
     * 
     * @var string
     */
    private $user_id;
    
    /**
     * Logger-instans
     * 
     * @var Bokfor_Ehandel_Logger|null
     */
    private $logger;
    
    /**
     * Cache time i sekunder
     * 
     * @var int
     */
    private $cache_time;
    
    /**
     * Konstruktor
     * 
     * @param string $api_url API base URL
     * @param string $api_key API Key för autentisering
     * @param string $user_id User ID för autentisering
     * @param Bokfor_Ehandel_Logger|null $logger Logger-instans
     * @param int $cache_time Cache time i sekunder (default: 300)
     */
    public function __construct($api_url, $api_key, $user_id, $logger = null, $cache_time = 300) {
        $this->api_url = rtrim($api_url, '/');
        $this->api_key = $api_key;
        $this->user_id = $user_id;
        $this->logger = $logger;
        $this->cache_time = $cache_time;
    }
    
    /**
     * Gör GET request
     * 
     * @param string $endpoint API endpoint (relativ URL)
     * @param array $args Ytterligare argument för wp_remote_get
     * @param bool $use_cache Om caching ska användas
     * @return array|WP_Error Response array eller WP_Error
     */
    public function get($endpoint, $args = array(), $use_cache = true) {
        $url = $this->api_url . '/' . ltrim($endpoint, '/');
        
        // Kontrollera cache
        if ($use_cache) {
            $cache_key = 'bokfore_api_' . md5($url . serialize($args));
            $cached = get_transient($cache_key);
            if ($cached !== false) {
                if ($this->logger) {
                    $this->logger->info('API response från cache: ' . $endpoint);
                }
                return $cached;
            }
        }
        
        // Bygg headers
        $headers = array(
            'Content-Type' => 'application/json',
            'X-API-Key' => $this->api_key,
            'X-User-ID' => $this->user_id
        );
        
        // Merge med custom headers
        if (isset($args['headers']) && is_array($args['headers'])) {
            $headers = array_merge($headers, $args['headers']);
        }
        
        // Bygg request args
        $request_args = array_merge(array(
            'timeout' => BOKFOREHANDEL_API_TIMEOUT,
            'headers' => $headers,
            'redirection' => 5,
            'httpversion' => '1.0',
            'blocking' => true
        ), $args);
        
        // Gör request
        $response = wp_remote_get($url, $request_args);
        
        // Hantera response
        $result = $this->handle_response($response, $endpoint);
        
        // Spara i cache om lyckad
        if ($use_cache && !is_wp_error($result) && isset($result['status_code']) && $result['status_code'] === 200) {
            set_transient($cache_key, $result, $this->cache_time);
        }
        
        return $result;
    }
    
    /**
     * Gör POST request
     * 
     * @param string $endpoint API endpoint (relativ URL)
     * @param array|string $data Data att skicka (array eller JSON-sträng)
     * @param array $args Ytterligare argument för wp_remote_post
     * @return array|WP_Error Response array eller WP_Error
     */
    public function post($endpoint, $data = array(), $args = array()) {
        $url = $this->api_url . '/' . ltrim($endpoint, '/');
        
        // Bygg headers
        $headers = array(
            'Content-Type' => 'application/json',
            'X-API-Key' => $this->api_key,
            'X-User-ID' => $this->user_id
        );
        
        // Merge med custom headers
        if (isset($args['headers']) && is_array($args['headers'])) {
            $headers = array_merge($headers, $args['headers']);
        }
        
        // Konvertera data till JSON om det är array
        $body = is_string($data) ? $data : json_encode($data);
        
        // Bygg request args
        $request_args = array_merge(array(
            'method' => 'POST',
            'timeout' => BOKFOREHANDEL_API_TIMEOUT,
            'headers' => $headers,
            'body' => $body,
            'redirection' => 5,
            'httpversion' => '1.0',
            'blocking' => true
        ), $args);
        
        // Gör request med retry-logik
        $response = $this->make_request_with_retry($url, $request_args, 'POST');
        
        return $this->handle_response($response, $endpoint);
    }
    
    /**
     * Gör PUT request
     * 
     * @param string $endpoint API endpoint (relativ URL)
     * @param array|string $data Data att skicka (array eller JSON-sträng)
     * @param array $args Ytterligare argument för wp_remote_request
     * @return array|WP_Error Response array eller WP_Error
     */
    public function put($endpoint, $data = array(), $args = array()) {
        $url = $this->api_url . '/' . ltrim($endpoint, '/');
        
        // Bygg headers
        $headers = array(
            'Content-Type' => 'application/json',
            'X-API-Key' => $this->api_key,
            'X-User-ID' => $this->user_id
        );
        
        // Merge med custom headers
        if (isset($args['headers']) && is_array($args['headers'])) {
            $headers = array_merge($headers, $args['headers']);
        }
        
        // Konvertera data till JSON om det är array
        $body = is_string($data) ? $data : json_encode($data);
        
        // Bygg request args
        $request_args = array_merge(array(
            'method' => 'PUT',
            'timeout' => BOKFOREHANDEL_API_TIMEOUT,
            'headers' => $headers,
            'body' => $body,
            'redirection' => 5,
            'httpversion' => '1.0',
            'blocking' => true
        ), $args);
        
        // Gör request med retry-logik
        $response = $this->make_request_with_retry($url, $request_args, 'PUT');
        
        return $this->handle_response($response, $endpoint);
    }
    
    /**
     * Hantera response från API-anrop
     * 
     * @param array|WP_Error $response Response från wp_remote_*
     * @param string $endpoint Endpoint som anropades
     * @return array Standardiserat response-array
     */
    private function handle_response($response, $endpoint) {
        // Hantera WP_Error
        if (is_wp_error($response)) {
            if ($this->logger) {
                $this->logger->error('API-fel för ' . $endpoint . ': ' . $response->get_error_message());
            }
            return array(
                'status_code' => 0,
                'body' => '',
                'data' => null,
                'error' => $response->get_error_message(),
                'is_wp_error' => true
            );
        }
        
        // Extrahera status och body
        $status_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body, true);
        
        // Logga om det är ett fel
        if ($status_code >= 400) {
            if ($this->logger) {
                $this->logger->warning('API returnerade felstatus ' . $status_code . ' för ' . $endpoint);
            }
        }
        
        return array(
            'status_code' => $status_code,
            'body' => $response_body,
            'data' => $data,
            'error' => null,
            'is_wp_error' => false
        );
    }
    
    /**
     * Gör request med retry-logik
     * 
     * @param string $url Fullständig URL
     * @param array $args Request arguments
     * @param string $method HTTP-metod ('POST', 'PUT', etc.)
     * @param bool $is_critical Om detta är ett kritiskt anrop
     * @return array|WP_Error Response
     */
    private function make_request_with_retry($url, $args, $method = 'POST', $is_critical = true) {
        // För kritiska anrop: 9 försök över 24h, för mindre kritiska: 3 försök över 1min
        $max_attempts = $is_critical ? 9 : 3;
        $intervals = $is_critical 
            ? array(1, 5, 15, 60, 300, 900, 3600, 21600, 86400) // 1s, 5s, 15s, 1m, 5m, 15m, 1h, 6h, 24h
            : array(1, 5, 15); // 1s, 5s, 15s
        
        $attempt = 0;
        $last_error = null;
        $last_response = null;
        
        while ($attempt < $max_attempts) {
            $attempt++;
            
            if ($this->logger) {
                $this->logger->info("API-anrop försök {$attempt}/{$max_attempts}: {$method} {$url}");
            }
            
            // Gör API-anrop
            if ($method === 'PUT') {
                $response = wp_remote_request($url, $args);
            } else {
                $response = wp_remote_post($url, $args);
            }
            
            $last_response = $response;
            
            // Kontrollera om retry behövs
            $should_retry = $this->should_retry_request($response, $attempt, $max_attempts);
            
            if (!$should_retry) {
                if ($attempt > 1 && $this->logger) {
                    $this->logger->info("API-anrop lyckades efter {$attempt} försök");
                }
                return $response; // Framgång eller permanent fel
            }
            
            // Spara fel för loggning
            if (is_wp_error($response)) {
                $last_error = $response->get_error_message();
                $error_code = $response->get_error_code();
                if ($this->logger) {
                    $this->logger->warning("API-anrop misslyckades (WP_Error): {$error_code} - {$last_error}");
                }
            } else {
                $status_code = wp_remote_retrieve_response_code($response);
                $last_error = "HTTP {$status_code}";
                if ($this->logger) {
                    $this->logger->warning("API-anrop misslyckades med statuskod: {$status_code}");
                }
            }
            
            // Vänta innan nästa försök (exponential backoff)
            if ($attempt < $max_attempts) {
                $delay = $intervals[$attempt - 1];
                
                // För längre delays (> 60 sekunder), använd WordPress cron istället för sleep
                if ($delay > 60) {
                    // Schemalägg nästa försök via WordPress cron
                    // Notera: Detta kräver att API-nycklar hämtas från options vid retry
                    if ($this->logger) {
                        $this->logger->info("Schemalagt retry-försök {$attempt} via WordPress cron");
                    }
                    // TODO: Implementera cron-baserad retry om nödvändigt
                    return $response; // Returnera nuvarande fel, retry kommer via cron
                } else {
                    sleep($delay);
                }
            }
        }
        
        // Alla försök misslyckades
        if ($this->logger) {
            $this->logger->error("API-anrop misslyckades efter {$max_attempts} försök. Sista fel: {$last_error}");
        }
        return $last_response; // Returnera sista försöket
    }
    
    /**
     * Avgör om ett API-anrop bör försökas igen
     * 
     * @param array|WP_Error $response Response från API-anropet
     * @param int $attempt Nuvarande försöksnummer
     * @param int $max_attempts Maximalt antal försök
     * @return bool Om retry bör göras
     */
    private function should_retry_request($response, $attempt, $max_attempts) {
        // Om vi har gjort alla försök, retry inte
        if ($attempt >= $max_attempts) {
            return false;
        }
        
        // Retry vid nätverksfel/timeout
        if (is_wp_error($response)) {
            $error_code = $response->get_error_code();
            // Retry vid timeout eller nätverksfel
            if (in_array($error_code, array('http_request_failed', 'timeout', 'connect_timeout'))) {
                return true;
            }
            return false; // Andra fel är permanenta
        }
        
        // Retry vid serverfel (5xx) eller rate limit (429)
        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code >= 500 && $status_code < 600) {
            return true; // Serverfel - försök igen
        }
        if ($status_code === 429) {
            return true; // Rate limit - försök igen med längre delay
        }
        
        // Inte retry vid klientfel (4xx) eller framgång (2xx)
        return false;
    }
}

