<?php
/**
 * Plugin Name: Bokför Ehandel
 * Plugin URI: https://bokforehandel.se/
 * Description: Bokför WooCommerce ordrar samt Klarna och Stripe utbetalningar till Bjorn Lunden.
 * Version: 1.2.4
 * Author: Ideäng Productions
 * Author URI: https://www.bokforehandel.se
 * Requires at least: 6.3
 * Tested up to: 6.7.2
 * Requires PHP: 7.4
 * WC requires at least: 8.0
 * WC tested up to: 9.7.1
 * Text Domain: bokfor-ehandel
 * Domain Path: /languages
 */

// Om denna fil anropas direkt, avbryt.
if (!defined('WPINC')) {
    die;
}

// Definiera plugin-konstanter
define('BOKFOREHANDEL_VERSION', '1.2.4');
define('BOKFOREHANDEL_PLUGIN_FILE', __FILE__);
define('BOKFOREHANDEL_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('BOKFOREHANDEL_PLUGIN_URL', plugin_dir_url(__FILE__));
define('BOKFOREHANDEL_API_URL', 'https://woo2bl.ideang.se');

// Performance and security constants
define('BOKFOREHANDEL_API_TIMEOUT', 10);
define('BOKFOREHANDEL_CACHE_TIME', 300); // 5 minutes
define('BOKFOREHANDEL_RATE_LIMIT', 5); // requests per minute
define('BOKFOREHANDEL_RATE_LIMIT_PERIOD', 60); // seconds

// Default account numbers
define('BOKFOREHANDEL_DEFAULT_SALES_ACCOUNT', '3001');
define('BOKFOREHANDEL_DEFAULT_VAT_ACCOUNT', '2610');
define('BOKFOREHANDEL_DEFAULT_BANK_ACCOUNT', '1930');
define('BOKFOREHANDEL_DEFAULT_RECEIVABLES_ACCOUNT', '1510');
define('BOKFOREHANDEL_DEFAULT_FEE_ACCOUNT', '6570');
define('BOKFOREHANDEL_DEFAULT_INPUT_VAT_ACCOUNT', '2640');

// Declare HPOS (High-Performance Order Storage) compatibility
add_action('before_woocommerce_init', function() {
    if (class_exists(\Automattic\WooCommerce\Utilities\FeaturesUtil::class)) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
    }
});

// Ladda textdomän för översättningar - WordPress 6.7+ kräver specifika hooks
// Enligt dokumentationen ska vi använda init hook med prioritet 0 (noll)
// https://make.wordpress.org/core/2024/02/29/handling-text-domain-loading-in-wordpress-6-7/
function bokforehandel_load_textdomain() {
    load_plugin_textdomain('bokforehandel', false, dirname(plugin_basename(__FILE__)) . '/languages/');
}

// Hookar översättningsladdningen till plugins_loaded för att säkerställa att den körs i rätt tidpunkt
add_action('plugins_loaded', 'bokforehandel_load_textdomain');

// Vänta med initiering tills efter att textdomäner är laddade (prioritet >0)
// Enligt nya rekommendationer för WordPress 6.7+ laddar vi textdomäner på 'init' med prioritet 0
// och sedan initierar pluginet på 'init' med prioritet 10
add_action('init', 'bokforehandel_init', 10); // Prioritet 10 för att köra efter textdomän-laddning

// Global variabel för att spåra om vi redan har initierat pluginet
$GLOBALS['bokforehandel_initialized'] = false;

/**
 * Huvudinitieringsfunktion - körs efter att alla textdomäner är korrekt laddade
 * Låt hela wp_loaded-hooken köras först för att undvika problem med textdomäner
 */
function bokforehandel_init() {
    // Kontrollera om pluginet redan har initialiserats för att undvika dubletter
    if ($GLOBALS['bokforehandel_initialized']) {
        return;
    }
    
    // Markera pluginet som initierat för att undvika dubletter
    $GLOBALS['bokforehandel_initialized'] = true;
    
    // Starta pluginet bara om WooCommerce finns
    if (class_exists('WC_Payment_Gateways')) {
        // Instansiera pluginet
        $GLOBALS['bokforehandel_instance'] = new Bokfor_Ehandel();
    } else {
        add_action('admin_notices', 'bokforehandel_wc_missing_notice');
    }
}

/**
 * Visa meddelande när WooCommerce saknas
 */

function bokforehandel_wc_missing_notice() {
    echo '<div class="error"><p>Bokför Ehandel kräver att WooCommerce är installerat och aktiverat.</p></div>';
}

class Bokfor_Ehandel {

    /**
     * Konstruktor - Minimal initialisering för att undvika för tidig laddning av textdomäner
     */
    public function __construct() {
        // Dela upp initieringen för att förhindra för tidig laddning av textdomäner
        
        // Registrera admin-menyer direkt - dessa måste köras tidigare än admin_init
        add_action('admin_menu', array($this, 'add_admin_menu'));
        
        // Steg 1: Lägg till en hook för att registrera övrig admin-funktionalitet (scripts, settings)
        // Använd admin_init istället för direkt registrering
        add_action('admin_init', array($this, 'register_admin_hooks'));
        
        // Steg 2: Lägg till en hook för att registrera AJAX-hanterare
        // Skjut upp till efter admin_init för att säkerställa att textdomäner är laddade
        add_action('admin_init', array($this, 'register_ajax_handlers'), 99);
        
        // Registrera cron-hook för retry-försök av API-anrop
        // ÄNDRAT: Lägg till hook för schemalagda retry-försök
        add_action('bokforehandel_retry_api_call', array($this, 'handle_scheduled_retry'), 10, 5);
        
        // Lägg till uppdateringskontroller
        add_filter('pre_set_site_transient_update_plugins', array($this, 'check_for_updates'));
        add_filter('plugins_api', array($this, 'plugin_info'), 10, 3);
        
        // WooCommerce-hooks hanteras nu på annat sätt
        // OBS: init_woocommerce_hooks har tagits bort eftersom den var tom och inte behövdes
    }
    
    /**
     * Registrera admin-hooks (menus, scripts, inställningar)
     * Körs under admin_init, efter att textdomäner är korrekt laddade
     */
    public function register_admin_hooks() {
        // Admin scripts (menyer hanteras nu direkt i konstruktorn)
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
        
        // Registrera inställningar
        $this->register_settings();
        
        // Registrera webhook endpoint (för integrationer med externa system)
        add_action('init', array($this, 'register_webhook_endpoint'), 15);
        
        // Lägg till omdirigering för gamla sidor
        add_action('current_screen', array($this, 'handle_admin_redirects'));
    }
    
    /**
     * Registrera AJAX-hanterare
     * Körs sent under admin_init för att undvika för tidig laddning av textdomäner
     */
    public function register_ajax_handlers() {
        // AJAX-hanterare för admin-funktioner
        add_action('wp_ajax_bokforehandel_save_settings', array($this, 'ajax_save_settings'));
        add_action('wp_ajax_bokforehandel_save_company', array($this, 'ajax_save_company'));
        add_action('wp_ajax_bokforehandel_save_accounting', array($this, 'ajax_save_accounting'));
        add_action('wp_ajax_bokforehandel_save_sync', array($this, 'ajax_save_sync'));
        add_action('wp_ajax_bokforehandel_test_connection', array($this, 'ajax_test_connection'));
        add_action('wp_ajax_bokforehandel_run_integration', array($this, 'ajax_run_integration'));
        add_action('wp_ajax_bokforehandel_get_runs', array($this, 'ajax_get_runs'));
        add_action('wp_ajax_bokforehandel_get_run_details', array($this, 'ajax_get_run_details'));
        add_action('wp_ajax_bokforehandel_check_status', array($this, 'ajax_check_status'));
        add_action('wp_ajax_bokforehandel_health_check', array($this, 'ajax_health_check'));
        // Removed network test function as it was only used for debugging
        // Removed org number check function as it was a simple utility not essential for core functionality
        // Removed ping function as it was only used for simple testing
        // Removed BL API test function as its functionality is covered by ajax_test_connection
        add_action('wp_ajax_bokforehandel_manual_sync', array($this, 'ajax_manual_sync'));
        add_action('wp_ajax_bokforehandel_run_retro_sync', array($this, 'ajax_run_retro_sync'));
        add_action('wp_ajax_bokforehandel_update_auto_sync', array($this, 'ajax_update_auto_sync'));
        add_action('wp_ajax_bokforehandel_get_synced_orders', array($this, 'ajax_get_synced_orders'));
        add_action('wp_ajax_bokforehandel_force_update_check', array($this, 'ajax_force_update_check'));
        add_action('wp_ajax_bokforehandel_clear_runs_log', array($this, 'ajax_clear_runs_log'));
        add_action('wp_ajax_bokforehandel_uninstall', array($this, 'ajax_uninstall_plugin'));
        add_action('wp_ajax_bokforehandel_reset_settings', array($this, 'ajax_reset_settings'));
        add_action('wp_ajax_bokforehandel_save_system_settings', array($this, 'ajax_save_system_settings'));
    }

    /**
     * Registrera WordPress-inställningar
     */
    public function register_settings() {
        // Registrera settings group först (WordPress 4.7+ kräver detta)
        register_setting(
            'bokforehandel_settings_group',
            'bokforehandel_settings_group',
            array(
                'type' => 'array',
                'description' => 'Bokför Ehandel Plugin Settings',
                'sanitize_callback' => function($input) {
                    return is_array($input) ? $input : array();
                },
                'default' => array()
            )
        );

        // Grundläggande inställningar
        register_setting('bokforehandel_settings_group', 'bokforehandel_company_name', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => ''
        ));
        register_setting('bokforehandel_settings_group', 'bokforehandel_store_name');
        register_setting('bokforehandel_settings_group', 'bokforehandel_organization_number');
        register_setting('bokforehandel_settings_group', 'bokforehandel_api_url');
        
        // BL API-inställningar
        register_setting('bokforehandel_settings_group', 'bokforehandel_bl_user_key');
        register_setting('bokforehandel_settings_group', 'bokforehandel_bl_client_secret');
        
        // Synkroniseringsinställningar
        register_setting('bokforehandel_settings_group', 'bokforehandel_sync_frequency');
        register_setting('bokforehandel_settings_group', 'bokforehandel_sync_enabled');
        register_setting('bokforehandel_settings_group', 'bokforehandel_sync_period');
        
        // Konto-inställningar
        register_setting('bokforehandel_settings_group', 'bokforehandel_accounts');
        
        // Betalningsmetoder
        register_setting('bokforehandel_settings_group', 'bokforehandel_payment_accounts');
        register_setting('bokforehandel_settings_group', 'bokforehandel_payment_settlements');
        
        // Klarna-inställningar
        register_setting('bokforehandel_settings_group', 'bokforehandel_klarna_enabled');
        register_setting('bokforehandel_settings_group', 'bokforehandel_klarna_api_username');
        register_setting('bokforehandel_settings_group', 'bokforehandel_klarna_api_password');
        
        // Stripe-inställningar
        register_setting('bokforehandel_settings_group', 'bokforehandel_stripe_enabled');
        register_setting('bokforehandel_settings_group', 'bokforehandel_stripe_api_key');
        
        // Dokumentuppladdning
        register_setting('bokforehandel_settings_group', 'bokforehandel_upload_receipts');
        register_setting('bokforehandel_settings_group', 'bokforehandel_upload_settlements');
        
        // Användare ID och integrationsstatus
        register_setting('bokforehandel_settings_group', 'bokforehandel_user_id');
        register_setting('bokforehandel_settings_group', 'bokforehandel_integration_status', function($value) {
            // Om inget värde har angetts (första aktiveringen), sätt till 'off'
            if ($value === false || $value === null || $value === '') {
                return 'off';
            }
            return $value;
        });
        
        // Faktureringsinformation
        register_setting('bokforehandel_settings_group', 'bokforehandel_billing_email');
        register_setting('bokforehandel_settings_group', 'bokforehandel_billing_reference');
        register_setting('bokforehandel_settings_group', 'bokforehandel_billing_address');
        register_setting('bokforehandel_settings_group', 'bokforehandel_billing_price_option');
        register_setting('bokforehandel_settings_group', 'bokforehandel_billing_agree');
        
        // Användaren har godkänt ansvarsfriskrivning
        register_setting('bokforehandel_settings_group', 'bokforehandel_disclaimer_agreed');
    }

    /**
     * Plugin-aktivering
     */
    public function activate() {
        
        // Skapa loggmapp om den inte finns
        $upload_dir = wp_upload_dir();
        $log_dir = $upload_dir['basedir'] . '/bokforehandel-logs';
        
        if (!file_exists($log_dir)) {
            wp_mkdir_p($log_dir);
        }
        
        // Skapa loggfil för att verifiera skrivrättigheter
        $test_log_file = $log_dir . '/plugin-activation.log';
        $log_test = @file_put_contents($test_log_file, 'Plugin aktiverad: ' . date('Y-m-d H:i:s') . "\n");
        
        if ($log_test === false) {
            add_action('admin_notices', function() {
                echo '<div class="notice notice-error is-dismissible"><p>Bokför Ehandel: Kunde inte skapa loggkatalog eller saknar skrivrättigheter till den. Kontrollera att WordPress har skrivrättigheter till <code>/wp-content/uploads</code>.</p></div>';
            });
        }
        
        // Skapa körningsloggtabell
        $this->ensure_runs_table_exists();
        
        // Registrera rewrite rules för webhook-endpoint
        $this->register_webhook_endpoint();
        flush_rewrite_rules(false);
        update_option('woo2bl_webhook_rules_updated', BOKFOREHANDEL_VERSION);
    }

    /**
     * Skapa körningsloggtabell om den inte finns
     */
    private function ensure_runs_table_exists() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'bokforehandel_runs';
        
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            run_id varchar(255) NOT NULL,
            user_id varchar(255) NOT NULL,
            operation varchar(100) NOT NULL,
            status varchar(50) NOT NULL,
            count int(11) DEFAULT 0,
            duration decimal(10,2) DEFAULT NULL,
            details longtext,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY run_id (run_id),
            KEY user_id (user_id),
            KEY created_at (created_at)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }

    /**
     * Avregistrera WP-Cron schemaläggning vid avaktivering
     */
    public function deactivate() {
        // Avregistrera daily_sync händelse
        // Vi använder inte längre WordPress Cron, server-side cron hanterar detta
        // $timestamp = wp_next_scheduled('bokforehandel_daily_sync');
        // if ($timestamp) {
        //     wp_unschedule_event($timestamp, 'bokforehandel_daily_sync');
        // }
    }

    /**
     * Lägg till admin-menyer
     */
    public function add_admin_menu() {
        // Huvudmeny - Ändra för att peka på company_settings_page istället
        add_menu_page(
            'Bokför Ehandel',
            'Bokför Ehandel',
            'manage_options',
            'bokforehandel-company',
            [$this, 'company_settings_page'],
            'dashicons-money-alt',
            58
        );
        
        // Undermenyer
        // Översiktssidan tas bort, Företagsinställningar blir huvudsidan
        
        // Företagsinställningar
        add_submenu_page(
            'bokforehandel-company',
            'Företagsinställningar',
            'Företagsinställningar',
            'manage_options',
            'bokforehandel-company',
            [$this, 'company_settings_page']
        );
        
        // Bokföringsinställningar
        add_submenu_page(
            'bokforehandel-company',
            'Bokföringsinställningar',
            'Bokföringsinställningar',
            'manage_options',
            'bokforehandel-accounting',
            [$this, 'accounting_settings_page']
        );
        
        // Synkroniseringssidan tas bort - dess funktionalitet finns i Körningshistorik
        
        // Körningshistorik
        add_submenu_page(
            'bokforehandel-company',
            'Synkronisering',
            'Synkronisering',
            'manage_options',
            'bokforehandel-runs',
            [$this, 'runs_page']
        );
        
        // Hjälpsida
        add_submenu_page(
            'bokforehandel-company',
            'Hjälp',
            'Hjälp',
            'manage_options',
            'bokforehandel-help',
            [$this, 'help_page']
        );
        
        // Systeminställningar
        add_submenu_page(
            'bokforehandel-company',
            'Systeminställningar',
            'Systeminställningar',
            'manage_options',
            'bokforehandel-system',
            [$this, 'system_settings_page']
        );
    }

    /**
     * Hantera omdirigering från gamla till nya sidhänvisningar
     * Detta hjälper användare som använder bokmärken eller följer gamla länkar
     */
    public function handle_admin_redirects() {
        $screen = get_current_screen();
        
        // Omdirigera från gamla översiktssidan till företagsinställningar
        if ($screen && $screen->id === 'toplevel_page_bokforehandel') {
            wp_redirect(admin_url('admin.php?page=bokforehandel-company'));
            exit;
        }
        
        // Omdirigera från synkroniseringssidan till körningshistorik
        if ($screen && $screen->id === 'bokfor-ehandel_page_bokforehandel-sync') {
            wp_redirect(admin_url('admin.php?page=bokforehandel-runs'));
            exit;
        }
    }

    /**
     * Ladda admin-stilar och skript
     */
    public function enqueue_admin_scripts($hook) {
        if (strpos($hook, 'bokforehandel') === false) {
            return;
        }
        
        // Sätt en flagga så vi vet att skripten har laddats
        if (!defined('BOKFOREHANDEL_SCRIPTS_LOADED')) {
            define('BOKFOREHANDEL_SCRIPTS_LOADED', true);
            
            // CSS
            wp_enqueue_style('bokforehandel-admin', plugins_url('assets/css/admin.css', __FILE__), array(), BOKFOREHANDEL_VERSION);
            
            // JavaScript
            wp_enqueue_script('jquery-ui-tabs');
            wp_enqueue_script('jquery-ui-datepicker'); // Lägg till datepicker explicit
            
            // Använd en unik version för att undvika cache-problem
            $unique_version = BOKFOREHANDEL_VERSION . '.' . time();
            wp_enqueue_script('bokforehandel-admin', plugins_url('assets/js/admin.js', __FILE__), array('jquery', 'jquery-ui-tabs', 'jquery-ui-datepicker'), $unique_version, true);
            
            // Lägg till AJAX-URL och nonce för säkerhet
            wp_localize_script('bokforehandel-admin', 'bokforehandel', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('bokforehandel_nonce'),
                'user_id' => get_option('bokforehandel_user_id', ''),
                'api_url' => BOKFOREHANDEL_API_URL,
                'integration_status' => get_option('bokforehandel_integration_status', false),
                'checking_status' => __('Kontrollerar integrationsstatus...', 'bokfor-ehandel'),
                'starting_sync' => __('Startar synkronisering...', 'bokfor-ehandel')
            ));
        }
        
        wp_enqueue_style('bokforehandel-admin-css', BOKFOREHANDEL_PLUGIN_URL . 'assets/css/admin.css', array(), BOKFOREHANDEL_VERSION);
        
        // Inlining CSS för toggle-switch
        $inline_css = '
        /* Toggle Switch */
        .switch {
            position: relative;
            display: inline-block;
            width: 50px;
            height: 24px;
            margin-right: 10px;
            vertical-align: middle;
        }
        .switch input {
            opacity: 0;
            width: 0;
            height: 0;
        }
        .slider {
            position: absolute;
            cursor: pointer;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: #ccc;
            transition: .4s;
        }
        .slider:before {
            position: absolute;
            content: "";
            height: 16px;
            width: 16px;
            left: 4px;
            bottom: 4px;
            background-color: white;
            transition: .4s;
        }
        input:checked + .slider {
            background-color: #2196F3;
        }
        input:focus + .slider {
            box-shadow: 0 0 1px #2196F3;
        }
        input:checked + .slider:before {
            transform: translateX(26px);
        }
        .slider.round {
            border-radius: 24px;
        }
        .slider.round:before {
            border-radius: 50%;
        }
        .status-text {
            display: inline-block;
            vertical-align: middle;
            font-weight: 600;
        }
        ';
        
        wp_add_inline_style('bokforehandel-admin-css', $inline_css);
    }

    /**
     * Hämta aktiva betalningsmetoder
     */
    public function get_active_payment_gateways() {
        if (!class_exists('WC_Payment_Gateways')) {
            return array();
        }
        
        $gateways = array();
        $wc_gateways = WC_Payment_Gateways::instance()->payment_gateways();
        
        foreach ($wc_gateways as $id => $gateway) {
            if ($gateway->enabled == 'yes') {
                $gateways[$id] = array(
                    'id' => $id,
                    'title' => $gateway->title,
                    'method' => $gateway->id,
                    'has_settlement' => in_array($gateway->id, array('kco', 'klarna_checkout', 'stripe'))
                );
            }
        }
        
        return $gateways;
    }

    /**
     * Översiktssidan (inaktiverad - omdirigeras nu till företagsinställningar)
     */
    public function settings_page() {
        // Omdirigera till företagsinställningar om javascript inte hanterar det
        echo '<script>window.location.href = "' . admin_url('admin.php?page=bokforehandel-company') . '";</script>';
        echo '<p>Omdirigerar till <a href="' . admin_url('admin.php?page=bokforehandel-company') . '">Företagsinställningar</a>...</p>';
        exit;
    }

    /**
     * Försök skapa WooCommerce API-nycklar automatiskt
     */
    public function maybe_create_wc_api_keys() {
        // Säkerställ att WooCommerce finns och vi har rätt behörigheter
        if (!function_exists('wc_get_logger') || !current_user_can('manage_woocommerce')) {
            return;
        }
        
        // Kolla om vi redan har API-nycklar
        if (!empty(get_option('bokforehandel_wc_consumer_key', ''))) {
            return;
        }
        
        // Endast tillgängligt i WooCommerce 3.0+
        if (!class_exists('WC_Auth')) {
            return;
        }
        
        try {
            global $wpdb;
            
            // Generera API-nycklar
            $description = 'Bokför Ehandel Integration';
            $user = wp_get_current_user();
            $permissions = 'read_write';
            
            // Skapa direkta databassökningar eftersom WC inte har ett officiellt API för detta
            $consumer_key = 'ck_' . wc_rand_hash();
            $consumer_secret = 'cs_' . wc_rand_hash();
            
            $wpdb->insert(
                $wpdb->prefix . 'woocommerce_api_keys',
                array(
                    'user_id' => $user->ID,
                    'description' => $description,
                    'permissions' => $permissions,
                    'consumer_key' => wc_api_hash($consumer_key),
                    'consumer_secret' => $consumer_secret,
                    'truncated_key' => substr($consumer_key, -7),
                ),
                array(
                    '%d',
                    '%s',
                    '%s',
                    '%s',
                    '%s',
                    '%s',
                )
            );
            
            if ($wpdb->insert_id) {
                update_option('bokforehandel_wc_consumer_key', $consumer_key);
                update_option('bokforehandel_wc_consumer_secret', $consumer_secret);
                update_option('bokforehandel_wc_api_url', site_url('wp-json/wc/v3'));
            }
        } catch (Exception $e) {
            // Logg eventuella fel
            if (function_exists('wc_get_logger')) {
                $logger = wc_get_logger();
                $logger->error('Kunde inte skapa WooCommerce API-nycklar: ' . $e->getMessage(), array('source' => 'bokforehandel'));
            }
        }
    }

    /**
     * Körningar-sidan
     */
    public function runs_page() {
        // Ladda mallen direkt istället för via wp_loaded
        include(BOKFOREHANDEL_PLUGIN_DIR . 'templates/runs.php');
    }

    /**
     * Hjälp-sidan
     */
    public function help_page() {
        // Ladda mallen direkt istället för via wp_loaded
        include(BOKFOREHANDEL_PLUGIN_DIR . 'templates/help.php');
    }

    /**
     * Systeminställningar-sidan
     */
    public function system_settings_page() {
        // Ladda mallen direkt istället för via wp_loaded
        include(BOKFOREHANDEL_PLUGIN_DIR . 'templates/system_settings.php');
    }

    /**
     * AJAX: Spara inställningar
     */
    public function ajax_save_settings() {
        try {
            // Rate limiting - max 5 requests per minute per user
            $user_id = get_current_user_id();
            $rate_limit_key = 'bokfore_rate_limit_' . $user_id;
            $requests = get_transient($rate_limit_key);

            if ($requests === false) {
                set_transient($rate_limit_key, 1, BOKFOREHANDEL_RATE_LIMIT_PERIOD);
            } elseif ($requests >= BOKFOREHANDEL_RATE_LIMIT) {
                wp_send_json_error('För många anrop. Vänta 1 minut.');
                exit;
            } else {
                set_transient($rate_limit_key, $requests + 1, BOKFOREHANDEL_RATE_LIMIT_PERIOD);
            }

            // Verifiera nonce för säkerhet
            if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'bokforehandel_nonce')) {
                wp_send_json_error('Säkerhetsverifiering misslyckades');
                exit;
            }

            // Kontrollera användarbehörighet
            if (!current_user_can('manage_options')) {
                wp_send_json_error('Du har inte behörighet');
                exit;
            }
    
        // Logga för felsökning (utan känslig data)
        $this->log('Försöker spara inställningar...', 'info');
        
        // Lista över fält att spara
        $fields = [
            'company_name' => 'sanitize_text_field',
            'store_name' => 'sanitize_text_field',
            'organization_number' => 'sanitize_text_field',
            'bl_user_key' => 'sanitize_text_field',
            'sync_enabled' => function($value) { return in_array($value, ['yes', 'true', '1']) ? 'yes' : 'no'; }
        ];
        
        // Lista över känsliga fält som inte ska loggas i klartext
        $sensitive_fields = ['bl_user_key', 'wc_consumer_key', 'wc_consumer_secret', 'klarna_api_username', 'klarna_api_password', 'stripe_api_key'];
    
        $success = true;
        $updated = 0;
        $unchanged = 0;
        global $wpdb;
        
        foreach ($fields as $field => $sanitize_func) {
            if (isset($_POST[$field])) {
                $value = call_user_func($sanitize_func, $_POST[$field]);
                $option_name = "bokforehandel_$field";
                
                // Hämta nuvarande värde för jämförelse
                $current_value = get_option($option_name, 'NOT_SET');
                
                // Logga endast om det inte är känslig data
                if (in_array($field, $sensitive_fields)) {
                    $current_display = !empty($current_value) && $current_value !== 'NOT_SET' ? '***' . substr($current_value, -4) : 'Ej satt';
                    $value_display = !empty($value) ? '***' . substr($value, -4) : 'Tom';
                    $this->log("Fält: $field, Nuvarande värde: '$current_display', Nytt värde: '$value_display'", 'info');
                } else {
                    $this->log("Fält: $field, Nuvarande värde: '$current_value', Nytt värde: '$value'", 'info');
                }
                
                // Jämför värdena
                if ($current_value === $value) {
                    $this->log("Värdet för $option_name är oförändrat - ingen uppdatering behövs", 'info');
                    $unchanged++;
                    continue; // Hoppa över update_option om värdena är identiska
                }
                
                // Försök spara med update_option
                $result = update_option($option_name, $value);
                
                if (!$result) {
                    $success = false;
                    $this->log("update_option för $option_name misslyckades", 'warning');
                    
                    // Försök med direkt DB-query om update_option misslyckas
                    $serialized_value = maybe_serialize($value);
                    
                    // Kontrollera först om optionen finns
                    $existing = $wpdb->get_var($wpdb->prepare(
                        "SELECT option_id FROM $wpdb->options WHERE option_name = %s",
                        $option_name
                    ));
                    
                    if ($existing) {
                        // UPDATE
                        $this->log("Försöker direkt DB-uppdatering för $option_name", 'info');
                        $db_result = $wpdb->query($wpdb->prepare(
                            "UPDATE $wpdb->options SET option_value = %s WHERE option_name = %s",
                            $serialized_value, $option_name
                        ));
                        if ($db_result === false) {
                            $this->log("DB-fel vid uppdatering: " . $wpdb->last_error, 'error');
                        } else {
                            $this->log("DB uppdatering lyckades: $db_result rad(er) påverkade", 'info');
                            if ($db_result > 0) {
                                $success = true; // Markera som lyckad om DB-uppdateringen fungerade
                                $updated++;
                            }
                        }
                    } else {
                        // INSERT
                        $this->log("Försöker direkt DB-infogning för $option_name", 'info');
                        $db_result = $wpdb->query($wpdb->prepare(
                            "INSERT INTO $wpdb->options (option_name, option_value, autoload) VALUES (%s, %s, %s)",
                            $option_name, $serialized_value, 'yes'
                        ));
                        if ($db_result === false) {
                            $this->log("DB-fel vid infogning: " . $wpdb->last_error, 'error');
                        } else {
                            $this->log("DB infogning lyckades: $db_result rad(er) påverkade", 'info');
                            if ($db_result > 0) {
                                $success = true; // Markera som lyckad om DB-infogningen fungerade
                                $updated++;
                            }
                        }
                    }
                } else {
                    $this->log("update_option för $option_name lyckades", 'info');
                    $updated++;
                }
            }
        }
    
        if ($success) {
            $message = "Inställningarna har verifierats.";
            if ($updated > 0) {
                $message = "Inställningarna har sparats. Uppdaterade fält: $updated";
            }
            if ($unchanged > 0) {
                $message .= " ($unchanged fält var redan korrekta)";
            }
            wp_send_json_success($message);
        } else {
            wp_send_json_error('Problem vid sparning av inställningar. Se WordPress felsökningsloggar för mer information.');
        }

        } catch (Exception $e) {
            error_log('Bokfor Ehandel: Fel i ajax_save_settings: ' . $e->getMessage());
            error_log('Bokfor Ehandel: Stack trace: ' . $e->getTraceAsString());
            wp_send_json_error('Ett oväntat fel uppstod. Felmeddelande har loggats för administratören.');
        }
    }

    /**
     * Registrera användare hos API-servern
     */
    private function register_user_with_api() {
        error_log('Bokforehandel: Försöker registrera användaren hos API-servern');
        
        $api_url = BOKFOREHANDEL_API_URL . '/api/users';
        error_log('Bokforehandel: API URL för registrering: ' . $api_url);
        
        // Förbered data att skicka
        $data = array(
            'organization_number' => get_option('bokforehandel_organization_number', ''),
            'company_name' => get_option('bokforehandel_company_name', get_bloginfo('name')),
            'store_name' => get_option('bokforehandel_store_name', get_bloginfo('name')),
            'email' => get_option('admin_email', ''),
            'bl_user_key' => get_option('bokforehandel_bl_user_key', ''),
            'wc_api_url' => get_option('bokforehandel_wc_api_url', site_url('wp-json/wc/v3')),
            'wc_consumer_key' => get_option('bokforehandel_wc_consumer_key', ''),
            'wc_consumer_secret' => get_option('bokforehandel_wc_consumer_secret', ''),
            'payment_accounts' => get_option('bokforehandel_payment_accounts', []),
            'payment_settlements' => get_option('bokforehandel_payment_settlements', []),
            'klarna_enabled' => get_option('bokforehandel_klarna_enabled', false),
            'klarna_api_username' => get_option('bokforehandel_klarna_api_username', ''),
            'klarna_api_password' => get_option('bokforehandel_klarna_api_password', ''),
            'stripe_enabled' => get_option('bokforehandel_stripe_enabled', false),
            'stripe_api_key' => get_option('bokforehandel_stripe_api_key', ''),
            'sync_frequency' => get_option('bokforehandel_sync_frequency', 'daily'),
            'sync_period' => get_option('bokforehandel_sync_period', 'lastday'),
            'upload_receipts' => get_option('bokforehandel_upload_receipts', false),
            'upload_settlements' => get_option('bokforehandel_upload_settlements', false)
        );
        
        // Lägg till konton
        $accounts = get_option('bokforehandel_accounts', []);
        $data = array_merge($data, $accounts);
        
        // Lägg till betalningsmetoder
        $data['payment_accounts'] = get_option('bokforehandel_payment_accounts', []);
        $data['payment_settlements'] = get_option('bokforehandel_payment_settlements', []);
        
        // Lägg till Klarna-inställningar om aktiverat
        if (get_option('bokforehandel_klarna_enabled', false)) {
            $data['klarna_api_username'] = get_option('bokforehandel_klarna_api_username', '');
            $data['klarna_api_password'] = get_option('bokforehandel_klarna_api_password', '');
            $data['klarna_enabled'] = true;
        }
        
        // Lägg till Stripe-inställningar om aktiverat
        if (get_option('bokforehandel_stripe_enabled', false)) {
            $data['stripe_api_key'] = get_option('bokforehandel_stripe_api_key', '');
            $data['stripe_enabled'] = true;
        }
        
        // Lägg till synkroniseringsinställningar
        $data['sync_frequency'] = get_option('bokforehandel_sync_frequency', 'daily');
        $data['sync_enabled'] = get_option('bokforehandel_sync_enabled', 'yes');
        $data['sync_period'] = get_option('bokforehandel_sync_period', 'lastday');
        
        // Hämta och lägg till butikens valuta från WooCommerce
        $data['store_currency'] = $this->get_store_currency();
        
        // Logga registreringsförsök
        error_log('Bokforehandel: Org.nr: ' . $data['organization_number'] . ', BL User Key: ' . (empty($data['bl_user_key']) ? 'saknas' : substr($data['bl_user_key'], 0, 5) . '...'));
        
        // Kontrollera nödvändiga fält
        if (empty($data['organization_number']) || empty($data['bl_user_key'])) {
            error_log('Bokforehandel: Registrering avbruten - saknar organisationsnummer eller BL User Key');
            return array(
                'status' => 'error',
                'message' => 'Organisationsnummer eller BL User Key saknas'
            );
        }
        
        // Kontrollera om det finns ett sparat användar-ID redan
        $existing_user_id = get_option('bokforehandel_user_id', '');
        if (!empty($existing_user_id)) {
            error_log('Bokforehandel: Användaren är redan registrerad med ID: ' . $existing_user_id . '. Uppdaterar istället.');
            return $this->update_user_with_api($existing_user_id);
        }
        
        // Skicka begäran till API-servern
        error_log('Bokforehandel: Skickar registreringsförfrågan till API-servern');
        
        $response = wp_remote_post($api_url, array(
            'method' => 'POST',
            'timeout' => 10,
            'redirection' => 5,
            'httpversion' => '1.0',
            'blocking' => true,
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-API-Key' => get_option('bokforehandel_bl_user_key')
            ),
            'body' => json_encode($data),
            'cookies' => array()
        ));
        
        // Kontrollera svaret
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            error_log('Bokforehandel: API-fel vid registrering: ' . $error_message);
            
            // Logg eventuella fel
            if (function_exists('wc_get_logger')) {
                $logger = wc_get_logger();
                $logger->error('API-fel vid registrering: ' . $error_message, array('source' => 'bokforehandel'));
            }
            return array(
                'status' => 'error',
                'message' => 'API-anslutningsfel: ' . $error_message
            );
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        // Logga API-svar
        error_log('Bokforehandel: API-svar statuskod: ' . $status_code);
        error_log('Bokforehandel: API-svar body: ' . wp_json_encode($body));
        
        if ($status_code === 200 && isset($body['status']) && $body['status'] === 'success' && isset($body['user_id'])) {
            // Spara användar-ID:et
            $user_id = sanitize_text_field($body['user_id']);
            update_option('bokforehandel_user_id', $user_id);
            error_log('Bokforehandel: Användar-ID sparades: ' . $user_id);

            // Hantera multibutiksläge om det finns
            $message = 'Användare registrerad';
            if (isset($body['multibutik_mode']) && $body['multibutik_mode'] === true) {
                $message = $body['message']; // Använd det detaljerade meddelandet från API:et
                error_log('Bokforehandel: Multibutiksläge upptäckt: ' . $message);
            }

            return array(
                'status' => 'success',
                'message' => $message,
                'multibutik_mode' => isset($body['multibutik_mode']) ? $body['multibutik_mode'] : false
            );
        } else if ($status_code === 400 && isset($body['message']) && (strpos($body['message'], 'finns redan') !== false || strpos($body['message'], 'är redan registrerad') !== false)) {
            // Butiken är redan registrerad - kontrollera om API:et returnerade ett user_id
            error_log('Bokforehandel: Butik finns redan, kontrollerar om user_id returnerades');

            if (isset($body['user_id'])) {
                // API:et returnerade ett giltigt user_id - använd detta direkt
                $user_id = sanitize_text_field($body['user_id']);
                update_option('bokforehandel_user_id', $user_id);
                error_log('Bokforehandel: Använde befintlig butik med returnerat ID: ' . $user_id);

                // Försök uppdatera befintlig butik med eventuella nya inställningar
                return $this->update_user_with_api($user_id);
            } else {
                // Fallback: Försök hitta användaren baserat på organisationsnummer OCH store_url för multibutik
                error_log('Bokforehandel: Inget user_id returnerat, försöker manuell sökning för multibutik');

                // Hämta alla användare och hitta den som matchar organisationsnumret och store_url
                $get_users_url = BOKFOREHANDEL_API_URL . '/api/users';
                $get_response = wp_remote_get($get_users_url, array(
                    'timeout' => 10,
                    'headers' => array(
                        'Content-Type' => 'application/json',
                        'X-API-Key' => get_option('bokforehandel_bl_user_key')
                    )
                ));

                if (!is_wp_error($get_response)) {
                    $get_status = wp_remote_retrieve_response_code($get_response);
                    $get_body = json_decode(wp_remote_retrieve_body($get_response), true);

                    if ($get_status === 200 && isset($get_body['status']) && $get_body['status'] === 'success' && isset($get_body['data'])) {
                        $org_nr = get_option('bokforehandel_organization_number', '');
                        // Extrahera store_url från wc_api_url
                        $wc_api_url = get_option('bokforehandel_wc_api_url', site_url('wp-json/wc/v3'));
                        $store_url = $this->extract_store_url($wc_api_url);

                        error_log('Bokforehandel: Söker efter användare med org_nr: ' . $org_nr . ' och store_url: ' . $store_url);

                        foreach ($get_body['data'] as $user) {
                            // Jämför både rensat och ej rensat organisationsnummer för flexibilitet
                            $user_org_clean = str_replace('-', '', $user['organization_number']);
                            $org_nr_clean = str_replace('-', '', $org_nr);

                            if ($user_org_clean === $org_nr_clean && isset($user['store_url']) && $user['store_url'] === $store_url) {
                                // Hittade exakt matchning för denna butik, spara ID och uppdatera
                                $user_id = sanitize_text_field($user['id']);
                                update_option('bokforehandel_user_id', $user_id);
                                error_log('Bokforehandel: Hittade befintlig butik via manuell sökning med ID: ' . $user_id);

                                // Uppdatera användaren
                                return $this->update_user_with_api($user_id);
                            }
                        }

                        // Om vi inte hittar exakt matchning, men hittar användare med samma org_nr,
                        // innebär det att detta är en ny butik för samma organisation
                        foreach ($get_body['data'] as $user) {
                            $user_org_clean = str_replace('-', '', $user['organization_number']);
                            $org_nr_clean = str_replace('-', '', $org_nr);

                            if ($user_org_clean === $org_nr_clean) {
                                error_log('Bokforehandel: Hittade befintlig organisation men ny butik. Org har ID: ' . $user['id'] . ', denna butik kommer att skapas som ny användare.');

                                // Detta är en ny butik för befintlig organisation.
                                // Felet beror på att vi försöker registrera en butik som redan finns i BL-systemet,
                                // men för multibutik ska vi kunna ha flera butiker för samma organisationsnummer.
                                // Returnera felet eftersom det är BL-systemet som säger att butiken är registrerad
                                return array(
                                    'status' => 'error',
                                    'message' => $body['message']
                                );
                            }
                        }
                    }
                }
            }

            error_log('Bokforehandel: Kunde inte hitta befintlig användare, trots att systemet säger att den finns');
            return array(
                'status' => 'error',
                'message' => 'Användare finns redan men kunde inte hittas'
            );
        } else {
            $error_msg = isset($body['message']) ? $body['message'] : 'Okänt fel';
            error_log('Bokforehandel: API-fel vid registrering: ' . $error_msg);
            
            // Logg eventuella fel
            if (function_exists('wc_get_logger')) {
                $logger = wc_get_logger();
                $logger->error('API-fel vid registrering: ' . $error_msg, array('source' => 'bokforehandel'));
            }
            return array(
                'status' => 'error',
                'message' => $error_msg
            );
        }
    }

    /**
     * Uppdatera användarinformation hos API-servern
     */
    private function update_user_with_api($user_id) {
        error_log('Bokforehandel: Uppdaterar användare med ID: ' . $user_id);
        
        if (empty($user_id)) {
            error_log('Bokforehandel: Kan inte uppdatera användare - användar-ID är tomt');
            return array(
                'status' => 'error',
                'message' => 'Användar-ID saknas'
            );
        }
        
        $api_url = BOKFOREHANDEL_API_URL . '/api/users/' . $user_id;
        error_log('Bokforehandel: API URL för uppdatering: ' . $api_url);
        
        // Konvertera boolean-värden från strängar till riktiga boolean
        $klarna_enabled = get_option('bokforehandel_klarna_enabled', false);
        $stripe_enabled = get_option('bokforehandel_stripe_enabled', false);
        $upload_receipts = get_option('bokforehandel_upload_receipts', false);
        $upload_settlements = get_option('bokforehandel_upload_settlements', false);
        
        // Konvertera till riktiga boolean-värden
        if ($klarna_enabled === '1' || $klarna_enabled === 'true' || $klarna_enabled === true) {
            $klarna_enabled = true;
        } else {
            $klarna_enabled = false;
        }
        
        if ($stripe_enabled === '1' || $stripe_enabled === 'true' || $stripe_enabled === true) {
            $stripe_enabled = true;
        } else {
            $stripe_enabled = false;
        }
        
        if ($upload_receipts === '1' || $upload_receipts === 'true' || $upload_receipts === true) {
            $upload_receipts = true;
        } else {
            $upload_receipts = false;
        }
        
        if ($upload_settlements === '1' || $upload_settlements === 'true' || $upload_settlements === true) {
            $upload_settlements = true;
        } else {
            $upload_settlements = false;
        }
        
        // Förbered data att skicka
        $data = array(
            'company_name' => get_option('bokforehandel_company_name', get_bloginfo('name')),
            'store_name' => get_option('bokforehandel_store_name', get_bloginfo('name')),
            'organization_number' => get_option('bokforehandel_organization_number', ''),
            'email' => get_option('admin_email', ''),
            'bl_user_key' => get_option('bokforehandel_bl_user_key', ''),
            'wc_api_url' => get_option('bokforehandel_wc_api_url', site_url('wp-json/wc/v3')),
            'wc_consumer_key' => get_option('bokforehandel_wc_consumer_key', ''),
            'wc_consumer_secret' => get_option('bokforehandel_wc_consumer_secret', ''),
            'payment_accounts' => get_option('bokforehandel_payment_accounts', []),
            'payment_settlements' => get_option('bokforehandel_payment_settlements', []),
            'klarna_enabled' => $klarna_enabled,
            'klarna_api_username' => get_option('bokforehandel_klarna_api_username', ''),
            'klarna_api_password' => get_option('bokforehandel_klarna_api_password', ''),
            'stripe_enabled' => $stripe_enabled,
            'stripe_api_key' => get_option('bokforehandel_stripe_api_key', ''),
            'sync_frequency' => get_option('bokforehandel_sync_frequency', 'daily'),
            'sync_period' => get_option('bokforehandel_sync_period', 'lastday'),
            'upload_receipts' => $upload_receipts,
            'upload_settlements' => $upload_settlements
        );
        
        // Lägg till konton
        $accounts = get_option('bokforehandel_accounts', []);
        $data = array_merge($data, $accounts);

        // Lägg till enskilda bokföringskonton (VIKTIGT för configurations-tabellen!)
        $data['sales_account'] = get_option('bokforehandel_sales_account', '3001');
        $data['vat_account'] = get_option('bokforehandel_vat_account', '2610');
        $data['bank_account'] = get_option('bokforehandel_bank_account', '1930');
        $data['receivables_account'] = get_option('bokforehandel_receivables_account', '1510');
        $data['fee_account'] = get_option('bokforehandel_fee_account', '6064');
        $data['input_vat_account'] = get_option('bokforehandel_input_vat_account', '2640');
        $data['journal_id'] = get_option('bokforehandel_journal_id', 'W');
        $data['klarna_journal_id'] = get_option('bokforehandel_klarna_journal_id', 'K');
        
        // Hämta och lägg till butikens valuta från WooCommerce
        $data['store_currency'] = $this->get_store_currency();
        $data['stripe_journal_id'] = get_option('bokforehandel_stripe_journal_id', 'S');

        // Lägg till enskilda bokföringskonton (VIKTIGT för configurations-tabellen!)
        $data['sales_account'] = get_option('bokforehandel_sales_account', '3001');
        $data['vat_account'] = get_option('bokforehandel_vat_account', '2610');
        $data['bank_account'] = get_option('bokforehandel_bank_account', '1930');
        $data['receivables_account'] = get_option('bokforehandel_receivables_account', '1510');
        $data['fee_account'] = get_option('bokforehandel_fee_account', '6064');
        $data['input_vat_account'] = get_option('bokforehandel_input_vat_account', '2640');
        $data['journal_id'] = get_option('bokforehandel_journal_id', 'W');
        $data['klarna_journal_id'] = get_option('bokforehandel_klarna_journal_id', 'K');
        
        // Hämta och lägg till butikens valuta från WooCommerce
        $data['store_currency'] = $this->get_store_currency();
        $data['stripe_journal_id'] = get_option('bokforehandel_stripe_journal_id', 'S');
        
        // Validera att alla viktiga konton finns och är korrekta
        $required_accounts = array(
            'bank_account' => 'Bankkonto',
            'receivables_account' => 'Kundfordringar',
            'fee_account' => 'Avgiftskonto',
            'sales_account' => 'Försäljningskonto',
            'vat_account' => 'Momskonto',
            'input_vat_account' => 'Ingående moms'
        );
        
        $missing_accounts = array();
        foreach ($required_accounts as $key => $label) {
            if (empty($data[$key]) || !is_numeric($data[$key])) {
                $missing_accounts[] = $label . ' (' . $key . ')';
            }
        }
        
        if (!empty($missing_accounts)) {
            $error_message = 'Följande konton saknas eller är ogiltiga: ' . implode(', ', $missing_accounts);
            error_log('Bokforehandel: ' . $error_message);
            error_log('Bokforehandel: Data som skulle skickas: ' . print_r($data, true));
            return array(
                'status' => 'error',
                'message' => $error_message
            );
        }
        
        // Logga vilka konton som skickas till API-servern (för felsökning)
        $account_values = array();
        foreach ($required_accounts as $key => $label) {
            $account_values[] = $label . ': ' . $data[$key];
        }
        error_log('Bokforehandel: Skickar följande konton till API-servern: ' . implode(', ', $account_values));
        
        // Kontrollera obligatoriska fält
        if (empty($data['organization_number']) || empty($data['bl_user_key'])) {
            error_log('Bokforehandel: Saknas obligatoriska fält för uppdatering - org. nummer eller BL User Key');
            return array(
                'status' => 'error',
                'message' => 'Saknas obligatoriska fält - organisationsnummer eller BL API-nyckel'
            );
        }
        
        // Lägg till betalningsmetoder
        $data['payment_accounts'] = get_option('bokforehandel_payment_accounts', []);
        $data['payment_settlements'] = get_option('bokforehandel_payment_settlements', []);
        
        // Lägg till Klarna-inställningar endast om aktiverat
        if ($klarna_enabled) {
            $data['klarna_api_username'] = get_option('bokforehandel_klarna_api_username', '');
            $data['klarna_api_password'] = get_option('bokforehandel_klarna_api_password', '');
        }
        
        // Lägg till Stripe-inställningar endast om aktiverat
        if ($stripe_enabled) {
            $data['stripe_api_key'] = get_option('bokforehandel_stripe_api_key', '');
        }
        
        // Lägg till synkroniseringsinställningar
        $data['sync_frequency'] = get_option('bokforehandel_sync_frequency', 'daily');
        
        // Konvertera sync_enabled till rätt format
        $sync_enabled = get_option('bokforehandel_sync_enabled', 'yes');
        if ($sync_enabled === 'yes' || $sync_enabled === true || $sync_enabled === 'true' || $sync_enabled === '1') {
            $data['sync_enabled'] = true;
        } else {
            $data['sync_enabled'] = false;
        }
        
        $data['sync_period'] = get_option('bokforehandel_sync_period', 'lastday');
        
        // Skicka begäran till API-servern
        error_log('Bokforehandel: Skickar uppdateringsförfrågan till API-servern');
        
        $response = wp_remote_request($api_url, array(
            'method' => 'PUT',
            'timeout' => 10,
            'redirection' => 5,
            'httpversion' => '1.0',
            'blocking' => true,
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-API-Key' => get_option('bokforehandel_bl_user_key')
            ),
            'body' => json_encode($data),
            'cookies' => array()
        ));
        
        // Kontrollera svaret
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            error_log('Bokforehandel: API-fel vid uppdatering: ' . $error_message);
            
            // Logg eventuella fel
            if (function_exists('wc_get_logger')) {
                $logger = wc_get_logger();
                $logger->error('API-fel vid uppdatering: ' . $error_message, array('source' => 'bokforehandel'));
            }
            
            return array(
                'status' => 'error',
                'message' => 'API-fel vid uppdatering: ' . $error_message
            );
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        error_log('Bokforehandel: API svarade med status: ' . $status_code);
        error_log('Bokforehandel: API svar: ' . $response_body);
        
        if ($status_code >= 200 && $status_code < 300) {
            try {
                $response_data = json_decode($response_body, true);
                
                if (isset($response_data['status']) && $response_data['status'] === 'success') {
                    error_log('Bokforehandel: Uppdatering lyckades');
                    return array(
                        'status' => 'success',
                        'message' => 'Uppdatering lyckades'
                    );
                } else {
                    $error_msg = isset($response_data['message']) ? $response_data['message'] : 'Okänt fel';
                    error_log('Bokforehandel: API returnerade felstatus: ' . $error_msg);
                    return array(
                        'status' => 'error',
                        'message' => $error_msg
                    );
                }
            } catch (Exception $e) {
                error_log('Bokforehandel: Fel vid tolkning av API-svar: ' . $e->getMessage());
                return array(
                    'status' => 'error',
                    'message' => 'Fel vid tolkning av API-svar: ' . $e->getMessage()
                );
            }
        } else {
            $error_message = 'Felmeddelande från API: ' . $status_code;
            error_log('Bokforehandel: ' . $error_message);
            return array(
                'status' => 'error',
                'message' => $error_message
            );
        }
    }

    /**
     * AJAX: Testa anslutningen till API-servern
     */
    public function ajax_test_connection() {
        // Verifiera nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'bokforehandel_nonce')) {
            wp_send_json_error('Säkerhetsverifiering misslyckades');
        }
        
        // Kontrollera användarrättigheter
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Du har inte behörighet att utföra denna åtgärd');
        }
        
        // Försök hämta användar-ID från anropshuvuden om tillgängligt
        $user_id = '';
        
        // Försök att hämta användar-ID från X-User-ID headern
        if (!empty($_SERVER['HTTP_X_USER_ID'])) {
            $user_id = sanitize_text_field($_SERVER['HTTP_X_USER_ID']);
            error_log('Bokforehandel: Användar-ID hittades i HTTP-header: ' . $user_id);
        }
        
        // Om inget användar-ID hittades i headern, använd det sparade värdet
        if (empty($user_id)) {
            $user_id = get_option('bokforehandel_user_id', '');
            error_log('Bokforehandel: Använder sparat användar-ID: ' . $user_id);
        }
        
        if (empty($user_id)) {
            error_log('Bokforehandel: Inget användar-ID hittades, kan inte testa anslutningen');
            wp_send_json_error('Du måste spara inställningarna först för att registrera dig hos tjänsten');
            return;
        }
        
        // Kontrollera anslutningen genom att hämta användardata
        $api_url = BOKFOREHANDEL_API_URL . '/api/users/' . $user_id;
        $this->log('Testar anslutning till API-server via URL: ' . $api_url, 'info');
        
        // ÄNDRAT: Använd BL User Key från inställningar istället för odefinierad konstant
        // BOKFOREHANDEL_API_KEY existerade inte - använd bokforehandel_bl_user_key istället
        $bl_user_key = get_option('bokforehandel_bl_user_key', '');
        if (empty($bl_user_key)) {
            wp_send_json_error('BL User Key saknas. Kontrollera inställningarna.');
            return;
        }
        
        // ÄNDRAT: API-servern använder @require_api_key som accepterar X-API-Key eller BL-User-Key
        // För /api/users/{user_id} behöver vi bara X-API-Key (inte trippel-autentisering)
        $headers = array(
            'Content-Type' => 'application/json',
            'X-API-Key' => $bl_user_key,  // Använd BL User Key från inställningar
            'X-User-ID' => $user_id        // Behåll för bakåtkompatibilitet om API-servern använder det
        );
        
        $this->log('Skickar anslutningstest med användar-ID: ' . $user_id, 'info');
        
        $response = wp_remote_get($api_url, array(
            'timeout' => 10,
            'redirection' => 5,
            'httpversion' => '1.0',
            'blocking' => true,
            'headers' => $headers
        ));
        
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            error_log('Bokforehandel: API-fel vid anslutningstest: ' . $error_message);
            wp_send_json_error('Kunde inte ansluta till API-servern: ' . $error_message);
            return;
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $body = json_decode($response_body, true);
        
        error_log('Bokforehandel: API-svar statuskod: ' . $status_code);
        error_log('Bokforehandel: API-svar body: ' . $response_body);
        
        if ($status_code === 200 && isset($body['status']) && $body['status'] === 'success') {
            error_log('Bokforehandel: Anslutningstest lyckades');
            wp_send_json_success('Anslutningen till API-servern fungerar');
        } else {
            $error_msg = isset($body['message']) ? $body['message'] : 'Okänt fel';
            error_log('Bokforehandel: Anslutningstest misslyckades: ' . $error_msg);
            wp_send_json_error('API-fel: ' . $error_msg);
        }
    }

    /**
     * Hantera statusändring för WooCommerce-order
     * Loggar endast information, faktisk synkronisering sker via cron
     * 
     * @param int $order_id Order ID
     * @param string $old_status Tidigare status
     * @param string $new_status Ny status
     */
    public function order_status_changed($order_id, $old_status, $new_status) {
        $this->log("Order #$order_id status ändrad från '$old_status' till '$new_status'", 'info');
    }
    
    /**
     * Hantera slutförd betalning för WooCommerce-order
     * Loggar endast information, faktisk synkronisering sker via cron
     * 
     * @param int $order_id Order ID
     */
    public function payment_complete($order_id) {
        // Hämta ordern för loggning
        $order = wc_get_order($order_id);
        if (!$order) {
            error_log("Bokfor Ehandel: Could not find order {$order_id}");
            return;
        }
        
        $payment_method = $order->get_payment_method();
        error_log("Bokfor Ehandel: Betalning slutförd för order {$order_id} med betalningsmetod {$payment_method} - kommer att synkroniseras vid nästa schemalagda körning");
    }
    
    // Removed single order synchronization functionality (add_order_meta_box, display_order_meta_box, ajax_sync_order)
    // as it was not supported by the backend API


     /**
     * Kontrollera om det redan finns en pågående eller nyligen slutförd körning för samma date_range och operation
     * 
     * @param string $date_range Datumintervall (t.ex. 'lastday', 'YYYYMMDD-YYYYMMDD')
     * @param string $operation Operationstyp (t.ex. 'all', 'woo', 'klarna')
     * @param int $user_id Användar-ID
     * @return array|false Returnerar array med körningsinfo om matchning hittas, annars false
     */
    private function check_existing_runs($date_range, $operation, $user_id) {
        try {
            // Hämta körningar från API
            $api_url = BOKFOREHANDEL_API_URL . '/api/runs/' . $user_id;
            $headers = [
                'Content-Type' => 'application/json',
                'X-API-Key' => get_option('bokforehandel_bl_user_key'),
                'X-User-ID' => $user_id
            ];
            
            $response = wp_remote_get($api_url, [
                'timeout' => 10,
                'headers' => $headers
            ]);
            
            if (is_wp_error($response)) {
                $this->log('Kunde inte hämta körningar för dubbelsynkningskontroll: ' . $response->get_error_message(), 'warning');
                return false; // Om vi inte kan hämta körningar, tillåt körningen att fortsätta
            }
            
            $status_code = wp_remote_retrieve_response_code($response);
            $body = json_decode(wp_remote_retrieve_body($response), true);
            
            if ($status_code >= 200 && $status_code < 300 && isset($body['status']) && $body['status'] === 'success' && isset($body['data'])) {
                $runs = $body['data'];
                $current_time = time();
                $five_minutes_ago = $current_time - (5 * 60); // 5 minuter i sekunder
                
                foreach ($runs as $run) {
                    // Kontrollera om körningen matchar operation och date_range
                    $run_operation = isset($run['operation']) ? $run['operation'] : '';
                    $run_date_range = null;
                    
                    // Hämta date_range från operations_data om det finns
                    if (isset($run['operations_data']) && !empty($run['operations_data'])) {
                        // operations_data kan vara en sträng (JSON) eller array
                        $ops_data = $run['operations_data'];
                        if (is_string($ops_data)) {
                            $ops_data = json_decode($ops_data, true);
                        }
                        if (is_array($ops_data) && isset($ops_data['date_range'])) {
                            $run_date_range = $ops_data['date_range'];
                        }
                    }
                    
                    // Kontrollera om operation matchar
                    if ($run_operation !== $operation) {
                        continue;
                    }
                    
                    // Kontrollera om date_range matchar
                    if ($run_date_range !== $date_range) {
                        continue;
                    }
                    
                    // Kontrollera status och tidpunkt
                    $run_status = isset($run['status']) ? $run['status'] : '';
                    $run_created_at = isset($run['created_at']) ? strtotime($run['created_at']) : 0;
                    
                    // Om körningen är pågående eller nyligen slutförd (inom 5 minuter)
                    if (($run_status === 'running') || 
                        ($run_status === 'completed' && $run_created_at >= $five_minutes_ago)) {
                        $this->log("Hittade matchande körning: ID {$run['id']}, Status: {$run_status}, Date range: {$date_range}, Operation: {$operation}", 'info');
                        return [
                            'id' => isset($run['id']) ? $run['id'] : null,
                            'status' => $run_status,
                            'created_at' => isset($run['created_at']) ? $run['created_at'] : null
                        ];
                    }
                }
            }
            
            return false;
        } catch (Exception $e) {
            $this->log('Fel vid kontroll av befintliga körningar: ' . $e->getMessage(), 'error');
            return false; // Om det blir fel, tillåt körningen att fortsätta
        }
    }

    /**
     * AJAX: Kör retroaktiv synkronisering
     */
    public function ajax_run_retro_sync() {
        // Kontrollera nonce
        $nonce_verified = false;
        
        // Kontrollera bokforehandel_nonce
        if (isset($_POST['nonce'])) {
            $nonce_verified = check_ajax_referer('bokforehandel_nonce', 'nonce', false);
        }
        
        // Om inte verifierad, kontrollera bokforehandel-run-retro-sync
        if (!$nonce_verified && isset($_POST['nonce'])) {
            $nonce_verified = check_ajax_referer('bokforehandel-run-retro-sync', 'nonce', false);
        }
        
        if (!$nonce_verified) {
            $this->log('Nonce-verifiering misslyckades vid retroaktiv synkronisering', 'error');
            wp_send_json_error(['message' => 'Säkerhetsverifiering misslyckades. Försök att ladda om sidan.']);
            return;
        }
        
        // Kontrollera att användaren har behörighet
        if (!current_user_can('manage_options')) {
            $this->log('Användare saknar behörighet för retroaktiv synkronisering', 'error');
            wp_send_json_error(['message' => 'Du har inte behörighet att utföra denna åtgärd.']);
            return;
        }
        
        // Hämta ID för den INLOGGADE användaren som klickade på knappen
        $current_wp_user_id = get_current_user_id();
        if (empty($current_wp_user_id)) {
            $this->log('Kunde inte hämta ID för inloggad användare vid retroaktiv synkronisering', 'error');
            wp_send_json_error(['message' => 'Kunde inte identifiera inloggad användare.']);
            return;
        }

        // Hämta BL User Key från inställningarna
        $bl_user_key = get_option('bokforehandel_bl_user_key');
        if (empty($bl_user_key)) {
            $this->log('BL User Key saknas i inställningarna vid retroaktiv synkronisering', 'error');
            wp_send_json_error(['message' => 'BL User Key saknas. Kontrollera dina API-inställningar.']);
            return;
        }

        // Hämta Organisationsnummer från inställningarna
        $organization_number = get_option('bokforehandel_organization_number');
        if (empty($organization_number)) {
            $this->log('Organisationsnummer saknas i inställningarna vid retroaktiv synkronisering', 'error');
            wp_send_json_error(['message' => 'Organisationsnummer saknas. Kontrollera dina API-inställningar.']);
            return;
        }
        
        // Hämta och validera input-data från POST
        $operation = isset($_POST['operation']) ? sanitize_text_field($_POST['operation']) : 'all';
        $start_date_input = isset($_POST['start_date']) ? sanitize_text_field($_POST['start_date']) : ''; // Format YYYYMMDD
        $end_date_input = isset($_POST['end_date']) ? sanitize_text_field($_POST['end_date']) : '';     // Format YYYYMMDD
        
        // Logga inkommande data
        $this->log("Retroaktiv synkronisering begärd av WP User ID: $current_wp_user_id. Operation: $operation, Start: $start_date_input, Slut: $end_date_input", 'info');
        
        // Formatera date_range för kontroll
        $date_range = !empty($start_date_input) && !empty($end_date_input) ? $start_date_input . '-' . $end_date_input : 'lastday';
        
        // Kontrollera om det redan finns en pågående eller nyligen slutförd körning för samma date_range och operation
        $user_id = get_option('bokforehandel_user_id', '');
        if (!empty($user_id)) {
            $existing_run = $this->check_existing_runs($date_range, $operation, $user_id);
            if ($existing_run) {
                $status_text = $existing_run['status'] === 'running' ? 'pågår redan' : 'slutfördes nyligen';
                $this->log("Förhindrar dubbelsynkning: En körning med samma date_range och operation {$status_text} (Körnings-ID: {$existing_run['id']})", 'info');
                wp_send_json_error([
                    'message' => "En synkronisering med samma datumintervall och operation {$status_text}. Vänta tills den är klar eller välj ett annat datumintervall.",
                    'existing_run_id' => $existing_run['id']
                ]);
                return;
            }
        }
        
        // Enkel validering av datumformat (ÅÅÅÅMMDD) - bara kontrollera att det är 8 siffror
        if (!empty($start_date_input) && !preg_match('/^\d{8}$/', $start_date_input)) {
            $this->log("Ogiltigt startdatum format för retroaktiv synkronisering: $start_date_input", 'error');
            wp_send_json_error(['message' => 'Ogiltigt startdatum. Använd formatet ÅÅÅÅMMDD (t.ex. 20250101).']);
            return;
        }
        
        if (!empty($end_date_input) && !preg_match('/^\d{8}$/', $end_date_input)) {
            $this->log("Ogiltigt slutdatum format för retroaktiv synkronisering: $end_date_input", 'error');
            wp_send_json_error(['message' => 'Ogiltigt slutdatum. Använd formatet ÅÅÅÅMMDD (t.ex. 20250131).']);
            return;
        }
        
        // Formatera datum för API-anrop
        $formatted_start_date = '';
        $formatted_end_date = '';
        
        if (!empty($start_date_input)) {
            $formatted_start_date = substr($start_date_input, 0, 4) . '-' . substr($start_date_input, 4, 2) . '-' . substr($start_date_input, 6, 2);
        }
        
        if (!empty($end_date_input)) {
            $formatted_end_date = substr($end_date_input, 0, 4) . '-' . substr($end_date_input, 4, 2) . '-' . substr($end_date_input, 6, 2);
        }
        
        // Förbered data för API-anrop
        $api_data = array(
            'operation' => $operation,
            'is_retroactive' => true
        );
        
        // Lägg till WooCommerce API URL om tillgänglig
        $wc_api_url = get_option('bokforehandel_wc_api_url', '');
        if (!empty($wc_api_url)) {
            $this->log("Retroaktiv synkronisering: Använder WooCommerce API URL: $wc_api_url", 'info');
            $api_data['wc_api_url'] = $wc_api_url;
        }
        
        // Lägg till datum
        if (!empty($formatted_start_date)) {
            $api_data['start_date'] = $formatted_start_date;
        }
        
        if (!empty($formatted_end_date)) {
            $api_data['end_date'] = $formatted_end_date;
        }
        
        // Lägg till WooCommerce API-nycklar
        $consumer_key = get_option('bokforehandel_wc_consumer_key', '');
        $consumer_secret = get_option('bokforehandel_wc_consumer_secret', '');
        
        if (!empty($consumer_key) && !empty($consumer_secret)) {
            $api_data['consumer_key'] = $consumer_key;
            $api_data['consumer_secret'] = $consumer_secret;
        }
        
        // Lägg till Klarna API-information
        $klarna_key = get_option('bokforehandel_klarna_api_username', '');
        $klarna_secret = get_option('bokforehandel_klarna_api_password', '');
        $klarna_enabled = get_option('bokforehandel_klarna_enabled', false);
        
        $api_data['klarna_enabled'] = $klarna_enabled;
        if (!empty($klarna_key) && !empty($klarna_secret)) {
            $api_data['klarna_api_username'] = $klarna_key;
            $api_data['klarna_api_password'] = $klarna_secret;
        }
        
        // Lägg till Stripe API-information
        $stripe_key = get_option('bokforehandel_stripe_api_key', '');
        $stripe_enabled = get_option('bokforehandel_stripe_enabled', false);
        
        $api_data['stripe_enabled'] = $stripe_enabled;
        if (!empty($stripe_key)) {
            $api_data['stripe_api_key'] = $stripe_key;
        }
        
        // Lägg till inställningar för bifogade dokument
        $api_data['upload_receipts'] = get_option('bokforehandel_upload_receipts', true) ? true : false;
        $api_data['upload_settlements'] = get_option('bokforehandel_upload_settlements', true) ? true : false;
        
        // Lägg till kontoinställningar
        $api_data['accounts'] = array(
            'bank_account' => get_option('bokforehandel_bank_account', '1930'),
            'receivables_account' => get_option('bokforehandel_receivables_account', '1580'),
            'fee_account' => get_option('bokforehandel_fee_account', '6064'),
            'input_vat_account' => get_option('bokforehandel_input_vat_account', '2640'),
            'sales_account' => get_option('bokforehandel_sales_account', '3001'),
            'vat_account' => get_option('bokforehandel_vat_account', '2610')
        );
        
        // Lägg till betalningskontoinställningar
        $payment_accounts = get_option('bokforehandel_payment_accounts', array());
        if (!empty($payment_accounts)) {
            $api_data['payment_accounts'] = $payment_accounts;
        }
        
        $this->log("Skickar retroaktiv synkronisering till API.", 'info');
        $this->log("Retroaktiv synkronisering: Operation: $operation", 'info');
        
        // Anropa API
        $api_url = BOKFOREHANDEL_API_URL . '/api/run-integration';
        
        // Hämta WC Consumer Key
        $wc_consumer_key = get_option('bokforehandel_wc_consumer_key', '');
        
        // Förbered body utan user_id
        $body_data = array(
            // 'user_id' => $current_wp_user_id, // Tas bort, identifiering sker via headers
            'operation' => $operation,
            'date_range' => !empty($start_date_input) && !empty($end_date_input) ? $start_date_input . '-' . $end_date_input : 'lastday',
            'operations_data' => $api_data
        );

        // ÄNDRAT: Använd retry-logik för kritiskt anrop
        // ÖKAD TIMEOUT: 180 sekunder (3 minuter) för att förhindra AJAX-fel vid hög serverbelastning
        $response = $this->make_api_request_with_retry($api_url, array(
            'method' => 'POST',
            'timeout' => 180,
            'httpversion' => '1.0',
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-API-Key' => $bl_user_key, // Använd hämtad BL User Key
                'X-Organization-Number' => $organization_number, // Lägg till organisationsnummer
                'X-WC-Consumer-Key' => $wc_consumer_key // Lägg till WC Consumer Key
            ),
            'body' => json_encode($body_data)
        ), 'POST', true);
        
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            $this->log("API-fel vid retroaktiv synkronisering efter retry-försök: $error_message", 'error');
            wp_send_json_error(['message' => 'Anslutningsfel: ' . $error_message]);
            return;
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $this->log("API-svar på retroaktiv synkronisering. Statuskod: $status_code, Svar: $body", 'info');
        
        $data = json_decode($body, true);
        
        if ($status_code >= 200 && $status_code < 300 && isset($data['status']) && $data['status'] === 'success') {
            // Framgångsrikt svar
            $run_id = isset($data['run_id']) ? intval($data['run_id']) : 0;
            $this->log("Retroaktiv synkronisering startad. Körnings-ID: $run_id", 'info');
            
            // Spara senaste run_id i databasen för att förenkla spårning
            update_option('bokforehandel_last_run_id', $run_id);
            
            // Använd WooCommerce-orderfelsökningslogik om det är en operation som berör ordrar
            if ($operation === 'all' || $operation === 'woo') {
                $this->log("Aktiverar detaljerad WooCommerce-orderfelsökning för run_id: $run_id", 'info');
                update_option('bokforehandel_debug_woo_orders', '1');
                
                // Om WooCommerce finns, logga information om datumintervallet
                if (class_exists('WooCommerce')) {
                    $this->log("Kontrollerar WooCommerce-ordrar inom datumintervallet: $formatted_start_date till $formatted_end_date", 'info');
                    // Anmärkning: debug_woo_orders_in_range-funktionen är borttagen då den inte var definierad
                }
            }
            
            // Skicka tillbaka run_id och annan relevant information
            wp_send_json_success([
                'message' => 'Synkronisering startad', 
                'run_id' => $run_id,
                'details_url' => admin_url('admin.php?page=bokforehandel-runs&view=details&run_id=' . $run_id)
            ]);
        } else {
            // Felaktigt svar
            $error_message = isset($data['message']) ? $data['message'] : 'Okänt fel';
            $this->log("Fel vid start av retroaktiv synkronisering: $error_message", 'error');
            
            // Försök extrahera mer detaljerad felinformation
            $error_details = '';
            if (isset($data['errors']) && is_array($data['errors'])) {
                foreach ($data['errors'] as $key => $error) {
                    $error_details .= "\n- " . $key . ": " . (is_array($error) ? implode(", ", $error) : $error);
                }
                $this->log("Detaljerade fel från API: $error_details", 'error');
            }
            
            wp_send_json_error(['message' => 'API-fel: ' . $error_message . $error_details]);
        }
    }


    /**
     * AJAX: Hämta körningar
     */
    public function ajax_get_runs() {
        // Kontrollera nonce - men fortsätt ändå med anropet om det misslyckas
        // Detta för att undvika felmeddelanden i UI
        $nonce_verified = false;
        
        // Kontrollera bokforehandel_nonce
        if (isset($_POST['nonce'])) {
            $nonce_verified = check_ajax_referer('bokforehandel_nonce', 'nonce', false);
        }
        
        // Om inte verifierad, kontrollera bokforehandel-get-runs
        if (!$nonce_verified && isset($_POST['nonce'])) {
            $nonce_verified = check_ajax_referer('bokforehandel-get-runs', 'nonce', false);
        }
        
        if (!$nonce_verified) {
            // Endast logga för debugging, inte för produktion
            // $this->log('Nonce-verifiering misslyckades vid hämtning av körningar', 'error');
        }
        
        // Hämta användar-ID från inställningar
        $user_id = get_option('bokforehandel_user_id', '');
        error_log('Bokforehandel: Användar-ID hämtat från inställningar i get_runs: ' . $user_id);
        
        if (empty($user_id)) {
            wp_send_json_success(['runs' => []]);
            return;
        }
        
        // Hämta runs från API med korrekt URL
        $api_url = BOKFOREHANDEL_API_URL . '/api/runs/' . $user_id;
        error_log("Bokforehandel: Hämtar körningar från: " . $api_url);
        error_log("Bokforehandel: Användar-ID: " . $user_id);
        
        $headers = [
            'Content-Type' => 'application/json',
            'X-API-Key' => get_option('bokforehandel_bl_user_key'),
            'X-User-ID' => $user_id
        ];
        
        $response = wp_remote_get($api_url, [
            'timeout' => 10,
            'headers' => $headers
        ]);
        
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            $this->log('Fel vid hämtning av körningar: ' . $error_message, 'error');
            wp_send_json_success(['runs' => []]);
            return;
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        // Logga API-svar för felsökning
        error_log('Bokforehandel: Statuskod från API (get_runs) - ' . $status_code);
        $response_body = wp_remote_retrieve_body($response);
        error_log('Bokforehandel: API-svar (get_runs) - ' . $response_body);
        
        if ($status_code >= 200 && $status_code < 300) {
            if (isset($body['status']) && $body['status'] === 'success') {
                // Bearbeta varje körning för att inkludera retroaktiva datumfält
                $runs = [];
                
                if (isset($body['data']) && is_array($body['data'])) {
                    $runs = $body['data'];
                    
                    foreach ($runs as &$run) {
                        // Markera om detta är en retroaktiv synkronisering
                        if (isset($run['operations_data']) && !empty($run['operations_data'])) {
                            // Kontrollera om detta är en retroaktiv synkronisering
                            if (isset($run['operations_data']['is_retroactive'])) {
                                $run['is_retroactive'] = true;
                                
                                // Om vi har sparade datum från retroaktiv synkronisering
                                if (isset($run['operations_data']['start_date']) && isset($run['operations_data']['end_date'])) {
                                    $run['saved_start_date'] = $run['operations_data']['start_date'];
                                    $run['saved_end_date'] = $run['operations_data']['end_date'];
                                }
                            }
                        }
                    }
                    unset($run); // Bryt referensen
                }
                
                wp_send_json_success(['runs' => $runs]);
            } else {
                if ($status_code === 200 && empty($body)) {
                    // API returnerade OK men ingen data
                    $this->log('API returnerade tom resultatlista', 'info');
                    wp_send_json_success(['runs' => []]);
                } else {
                    // Okänt fel i API-svaret
                    $error_msg = isset($body['message']) ? $body['message'] : 'Tjänsten returnerade ingen data';
                    $this->log('API returnerade felaktigt format: ' . $error_msg, 'warning');
                    wp_send_json_success(['runs' => []]);
                }
            }
        } else {
            // API-anropet gav en felkod
            $error_msg = isset($body['message']) ? $body['message'] : 'Fel vid hämtning av körningar (HTTP ' . $status_code . ')';
            $this->log('API-fel vid hämtning av körningar: ' . $error_msg, 'error');
            wp_send_json_success(['runs' => []]);
        }
    }

    /**
     * AJAX: Kontrollera integrationsstatus
     */
    public function ajax_check_status() {
        if (!$this->verify_ajax_security('bokforehandel_nonce')) {
            return;
        }
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Du har inte behörighet att utföra denna åtgärd'));
            return;
        }
        
        $user_id = get_option('bokforehandel_user_id', '');
        if (empty($user_id)) {
            wp_send_json_error(array('message' => 'Du måste spara inställningarna först för att registrera dig hos tjänsten'));
            return;
        }
        
        // Hämta BL User Key
        $bl_user_key = get_option('bokforehandel_bl_user_key', '');
        if (empty($bl_user_key)) {
            wp_send_json_error(array('message' => 'BL User Key saknas. Fyll i det i inställningarna.'));
            return;
        }
        
        // Hämta organisationsnummer
        $org_number = get_option('bokforehandel_organization_number', '');
        if (empty($org_number)) {
            wp_send_json_error(array('message' => 'Organisationsnummer saknas. Fyll i det i inställningarna.'));
            return;
        }
        
        // Rensa organisationsnumret från bindestreck
        $org_number_clean = str_replace('-', '', $org_number);
        
        // Använd vår gemensamma funktion för att kontrollera integrationsstatus
        $integration_status = $this->check_bl_integration_status($user_id);
        
        if ($integration_status === 'error') {
            wp_send_json_error(array(
                'message' => 'Inaktiv. Du måste aktivera integrationen hos Bjorn Lunden för att kunna använda Bokför Ehandel. Aktivera integrationen hos Bjorn Lunden, fyll därefter i publik nyckel och organisationsnummer för att aktivera tillägget.'
            ));
            return;
        }
        
        wp_send_json_success(array(
            'integration_status' => $integration_status,
            'message' => 'Integration status: ' . $integration_status
        ));
    }
    
    /**
     * Kontrollera BL-integrationsstatus
     *
     * @param string $user_id Användar-ID
     * @return string Status ('on', 'off', eller 'error')
     */
    private function check_bl_integration_status($user_id) {
        try {
            // Logga för felsökning
            error_log('Bokforehandel: Kontrollerar BL-integrationsstatus för användar-ID: ' . $user_id);

            // Check cache first (5 minute cache)
            $cache_key = 'bokforeintegration_status_' . $user_id;
            $cached_status = get_transient($cache_key);
            if ($cached_status !== false) {
                error_log('Bokfor Ehandel: Using cached integration status: ' . $cached_status);
                return $cached_status;
            }

            // Kontrollera lokalt sparad integrationsstatus
            $local_integration_status = get_option('bokforehandel_integration_status', 'not_configured');

            // Vi fortsätter med kontroll mot API även om lokal status är not_configured
            // Gamla installationer kanske inte har detta värde satt
            if ($local_integration_status === 'off') {
                error_log('Bokforhandel: Lokal integrationsstatus är off. Returnerar.');
                set_transient($cache_key, 'off', 300); // Cache for 5 minutes
                return 'off';
            }
        
        // Skapa API-URL för att kontrollera integrationsstatus
        $api_url = BOKFOREHANDEL_API_URL . '/api/integration_status/' . $user_id;
        
        // Hämta BL User Key
        $bl_user_key = get_option('bokforehandel_bl_user_key', '');
        if (empty($bl_user_key)) {
            error_log('Bokforehandel: BL User Key saknas');
            return 'off';
        }
        
        // Hämta organisationsnummer
        $org_number = get_option('bokforehandel_organization_number', '');
        if (empty($org_number)) {
            error_log('Bokforehandel: Organisationsnummer saknas');
            return 'off';
        }
        
        // Rensa organisationsnumret från bindestreck
        $org_number_clean = str_replace('-', '', $org_number);
        
        // Kontrollera anslutningen genom att anropa integration_status API
        // ÄNDRAT: API-servern använder @require_api_key som kräver X-API-Key header
        // Tidigare: 'BL-User-Key' och 'Organization-Number' (fel header-namn)
        // Nu: 'X-API-Key' (korrekt enligt API-serverns @require_api_key-dekorator)
        $api_url = BOKFOREHANDEL_API_URL . '/api/integration_status/' . $user_id;
        
        $this->log('Testar BL API-anslutning för användar-ID: ' . $user_id . ', org nr: ' . $org_number, 'info');
        
        $response = wp_remote_get($api_url, array(
            'timeout' => 10,
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-API-Key' => $bl_user_key  // ÄNDRAT: Använd X-API-Key istället för BL-User-Key
                // OBS: Organization-Number och BL-User-Key kan användas av API-servern för bakåtkompatibilitet,
                // men primär autentisering sker via X-API-Key enligt @require_api_key-dekoratorn
            )
        ));
        
        // Hantera fel
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            error_log('Bokforehandel: Fel vid kontroll av integrationsstatus: ' . $error_message);
            return 'error';
        }
        
        // Tolka API-svar
        $status_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        error_log('Bokforehandel: API-svar statuskod: ' . $status_code);
        $response_body = wp_remote_retrieve_body($response);
        error_log('Bokforehandel: API-svar body: ' . $response_body);
        
        // Kontrollera att anropet lyckades
        if ($status_code === 200 && isset($body['status']) && $body['status'] === 'success') {
            $api_integration_status = isset($body['integration_status']) ? $body['integration_status'] : 'off';
            
            // Om lokal status är explicit avstängd, avbryt direkt
            if ($local_integration_status === 'off') {
                return 'off';
            }
            
            // Om lokal status är 'not_configured' och API säger 'on', använd API-svaret
            if ($local_integration_status === 'not_configured') {
                $final_status = $api_integration_status;
                error_log('Bokforehandel: Ingen lokal integrationsstatus hittades. Använder API-status: ' . $api_integration_status);
                
                // Om API säger 'on', uppdatera den lokala statusen för framtida användning
                if ($api_integration_status === 'on') {
                    update_option('bokforehandel_integration_status', 'on');
                    error_log('Bokforehandel: Uppdaterade lokal integrationsstatus till on baserat på API-svar');
                }
            } else {
                // Annars, använd både API och lokal status
                $final_status = ($api_integration_status === 'on' && $local_integration_status === 'on') ? 'on' : 'off';
            }
            
            error_log('Bokforehandel: API integrationsstatus: ' . $api_integration_status);
            error_log('Bokforehandel: Lokal integrationsstatus: ' . $local_integration_status);
            error_log('Bokforehandel: Final integrationsstatus: ' . $final_status);
            
            return $final_status;
        } else {
            $error_msg = isset($body['message']) ? $body['message'] : 'Okänt fel';
            error_log('Bokforehandel: API-fel vid kontroll av integrationsstatus: ' . $error_msg);
            return 'error';
        }
        } catch (Exception $e) {
            error_log('Bokforehandel: Exception vid kontroll av integrationsstatus: ' . $e->getMessage());
            return 'error';
        }
    }
    
    // Removed ajax_check_org_number function as it was a simple utility that just returned
    // the organization number with and without hyphens, not essential for core functionality
    
    // Removed ajax_ping function as it was only used for simple testing and is not necessary for core functionality
    
    // Removed ajax_test_bl_api function as its functionality is largely covered by the
    // more comprehensive ajax_test_connection function which also tests the BL API connection
    
    // Removed ajax_test_network function as it was only used for debugging purposes and
    // its functionality is largely covered by the more comprehensive ajax_test_connection function
    
    /**
     * AJAX: Hämta detaljerad information om en körning
     */
    public function ajax_get_run_details() {
        // Kontrollera nonce - men gör det valfritt för att förhindra fel vid automatiska uppdateringar
        // Detta tillhandahåller fortfarande säkerhet men undviker fel när anropet görs automatiskt
        $nonce_verified = false;
        
        // Kontrollera bokforehandel_nonce
        if (isset($_POST['nonce'])) {
            $nonce_verified = check_ajax_referer('bokforehandel_nonce', 'nonce', false);
        }
        
        // Om inte verifierad, kontrollera bokforehandel-get-run-details
        if (!$nonce_verified && isset($_POST['nonce'])) {
            $nonce_verified = check_ajax_referer('bokforehandel-get-run-details', 'nonce', false);
        }
        
        if (!$nonce_verified && is_user_logged_in() && !defined('DOING_AJAX_AUTOFETCH')) {
            // Logga bara felet om det är en direkt användarinitierad begäran
            $this->log('Nonce-verifiering misslyckades vid hämtning av körningsdetaljer', 'error');
        }
        
        if (!current_user_can('manage_options')) {
            $this->log('Användare saknar behörighet för att hämta körningsdetaljer', 'error');
            wp_send_json_error(['message' => 'Du har inte behörighet att utföra denna åtgärd']);
            return;
        }
        
        // Acceptera run_id från både GET och POST beroende på hur anropet görs
        $run_id = 0;
        if (isset($_POST['run_id'])) {
            $run_id = intval($_POST['run_id']);
        } elseif (isset($_GET['run_id'])) {
            $run_id = intval($_GET['run_id']);
        }
        
        if ($run_id <= 0) {
            // Försök extrahera run_id från refererande URL om tillgänglig
            $referer = isset($_SERVER['HTTP_REFERER']) ? $_SERVER['HTTP_REFERER'] : '';
            if (preg_match('/[&?]run_id=(\d+)/', $referer, $matches)) {
                $run_id = intval($matches[1]);
            }
            
            // Om vi fortfarande inte har ett run_id, använd det från URL-sökvägen
            if ($run_id <= 0) {
                $request_uri = isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : '';
                if (preg_match('#/run/(\d+)/details#', $request_uri, $matches)) {
                    $run_id = intval($matches[1]);
                }
            }
            
            if ($run_id <= 0) {
                $this->log('Ogiltigt körnings-ID vid hämtning av detaljer, kunde inte fastställa ID', 'error');
                wp_send_json_error(['message' => 'Ogiltigt körnings-ID']);
                return;
            }
        }
        
        $user_id = get_option('bokforehandel_user_id', '');
        if (empty($user_id)) {
            $this->log('Användar-ID saknas vid hämtning av körningsdetaljer', 'error');
            wp_send_json_error(['message' => 'Användar-ID saknas. Kontrollera inställningarna.']);
            return;
        }
        
        // Gör API-anrop till korrekt endpoint med API-nyckel
        $api_url = BOKFOREHANDEL_API_URL . '/api/run/' . $run_id . '/details';
        $this->log('Hämtar körningsdetaljer från: ' . $api_url, 'info');
        
        $response = wp_remote_get($api_url, [
            'timeout' => 10,
            'headers' => [
                'Content-Type' => 'application/json',
                'X-API-Key' => get_option('bokforehandel_bl_user_key'),
                'X-User-ID' => $user_id
            ]
        ]);
        
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            $this->log('API-fel vid hämtning av körningsdetaljer: ' . $error_message, 'error');
            wp_send_json_error(['message' => 'Anslutningsfel: ' . $error_message]);
            return;
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $this->log('API-svar för körningsdetaljer. Statuskod: ' . $status_code . ', Svar: ' . $body, 'info');
        
        $data = json_decode($body, true);
        
        if ($status_code === 200 && isset($data['status']) && $data['status'] === 'success') {
            wp_send_json_success($data['data']);
        } else {
            $error_msg = isset($data['message']) ? $data['message'] : 'Okänt fel';
            $this->log('API-fel vid hämtning av körningsdetaljer: ' . $error_msg, 'error');
            wp_send_json_error(['message' => 'API-fel: ' . $error_msg]);
        }
    }
    
    /**
     * AJAX: Kör integration
     */
    public function ajax_run_integration() {
        // Verifiera nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'bokforehandel_nonce')) {
            wp_send_json_error('Säkerhetsverifiering misslyckades');
        }
        
        // Kontrollera användarrättigheter
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Du har inte behörighet att utföra denna åtgärd');
        }
        
        // Kontrollera att vi har nödvändiga parametrar
        if (!isset($_POST['operation']) || empty($_POST['operation'])) {
            wp_send_json_error('Operationstyp saknas');
            return;
        }
        
        if (!isset($_POST['date_range']) || empty($_POST['date_range'])) {
            wp_send_json_error('Datumintervall saknas');
            return;
        }
        
        $operation = sanitize_text_field($_POST['operation']);
        $date_range = sanitize_text_field($_POST['date_range']);
        
        // Hämta användar-ID
        $user_id = get_option('bokforehandel_user_id', '');
        if (empty($user_id)) {
            wp_send_json_error('Du måste spara inställningarna först för att registrera dig hos tjänsten');
            return;
        }
        
        // Kontrollera om det redan finns en pågående eller nyligen slutförd körning för samma date_range och operation
        $existing_run = $this->check_existing_runs($date_range, $operation, $user_id);
        if ($existing_run) {
            $status_text = $existing_run['status'] === 'running' ? 'pågår redan' : 'slutfördes nyligen';
            $this->log("Förhindrar dubbelsynkning: En körning med samma date_range och operation {$status_text} (Körnings-ID: {$existing_run['id']})", 'info');
            wp_send_json_error("En synkronisering med samma datumintervall och operation {$status_text}. Vänta tills den är klar eller välj ett annat datumintervall.");
            return;
        }
        
        // Kontrollera BL-integrationsstatus innan synkronisering
        $integration_status = $this->check_bl_integration_status($user_id);
        if ($integration_status !== 'on') {
            wp_send_json_error('Aktivera integration i Bjorn Lunden innan synkning kan ske.');
            return;
        }
        
        // Förbered data som behövs för integrationen
        $operations_data = array();
        
        // Lägg till butiksinformation
        $operations_data['store_name'] = get_option('bokforehandel_store_name', get_bloginfo('name'));
        $operations_data['company_name'] = get_option('bokforehandel_company_name', '');
        $operations_data['organization_number'] = get_option('bokforehandel_organization_number', '');
        
        // Lägg till inställningar för bifogade dokument
        $operations_data['upload_receipts'] = get_option('bokforehandel_upload_receipts', true) ? true : false;
        $operations_data['upload_settlements'] = get_option('bokforehandel_upload_settlements', true) ? true : false;
        $operations_data['journal_id'] = get_option('bokforehandel_journal_id', 'W');
        $operations_data['klarna_journal_id'] = get_option('bokforehandel_klarna_journal_id', 'W');
        $operations_data['stripe_journal_id'] = get_option('bokforehandel_stripe_journal_id', 'W');
        
        // Lägg till kontoinställningar
        $operations_data['accounts'] = array(
            'bank_account' => get_option('bokforehandel_bank_account', '1930'),
            'receivables_account' => get_option('bokforehandel_receivables_account', '1580'),
            'fee_account' => get_option('bokforehandel_fee_account', '6064'),
            'input_vat_account' => get_option('bokforehandel_input_vat_account', '2640'),
            'sales_account' => get_option('bokforehandel_sales_account', '3001'),
            'vat_account' => get_option('bokforehandel_vat_account', '2610')
        );
        
        // Lägg till betalningskontoinställningar
        $payment_accounts = get_option('bokforehandel_payment_accounts', array());
        if (!empty($payment_accounts)) {
            $operations_data['payment_accounts'] = $payment_accounts;
        }
        
        // Lägg till WooCommerce API-information
        $operations_data['woocommerce'] = array(
            'api_url' => get_option('bokforehandel_wc_api_url', ''),
            'consumer_key' => get_option('bokforehandel_wc_consumer_key', ''),
            'consumer_secret' => get_option('bokforehandel_wc_consumer_secret', '')
        );
        
        // Lägg till Klarna API-information
        $klarna_key = get_option('bokforehandel_klarna_api_username', '');
        $klarna_secret = get_option('bokforehandel_klarna_api_password', '');
        $klarna_enabled = get_option('bokforehandel_klarna_enabled', false);
        
        $operations_data['klarna_enabled'] = $klarna_enabled;
        if (!empty($klarna_key) && !empty($klarna_secret)) {
            $operations_data['klarna_api_username'] = $klarna_key;
            $operations_data['klarna_api_password'] = $klarna_secret;
        }
        
        // Lägg till Stripe API-information
        $stripe_key = get_option('bokforehandel_stripe_api_key', '');
        $stripe_enabled = get_option('bokforehandel_stripe_enabled', false);
        
        $operations_data['stripe_enabled'] = $stripe_enabled;
        if (!empty($stripe_key)) {
            $operations_data['stripe_api_key'] = $stripe_key;
        }
        
        // ÄNDRAT: Använd /api/run-integration istället för /api/run/{user_id}
        // API-servern kräver trippel-autentisering (X-API-Key, X-Organization-Number, X-WC-Consumer-Key)
        // via @require_wp_auth-dekoratorn för säkerhet
        $api_url = BOKFOREHANDEL_API_URL . '/api/run-integration';
        
        // Hämta nödvändiga värden för autentisering
        $bl_user_key = get_option('bokforehandel_bl_user_key', '');
        $organization_number = get_option('bokforehandel_organization_number', '');
        $wc_consumer_key = get_option('bokforehandel_wc_consumer_key', '');
        
        // Validera att alla nödvändiga värden finns
        if (empty($bl_user_key) || empty($organization_number) || empty($wc_consumer_key)) {
            wp_send_json_error('Saknar nödvändiga API-nycklar eller organisationsnummer. Kontrollera inställningarna.');
            return;
        }
        
        // Skapa data för API-anropet
        // OBS: user_id skickas INTE i body längre - API-servern identifierar användaren via headers
        $api_data = array(
            'operation' => $operation, // ÄNDRAT: 'operation' istället för 'operation_type' för kompatibilitet med API-servern
            'date_range' => $date_range,
            'operations_data' => $operations_data
        );
        
        // Logga exakt vad som skickas till API:et för felsökning
        $this->log('Skickar integration-begäran till API: ' . $api_url, 'info');
        $this->log('Operation: ' . $operation . ', Date range: ' . $date_range, 'info');
        
        // ÄNDRAT: Använd retry-logik för kritiskt anrop
        // API-servern validerar användaren baserat på kombinationen av dessa tre värden
        // ÖKAD TIMEOUT: 180 sekunder (3 minuter) för att förhindra AJAX-fel vid hög serverbelastning
        $response = $this->make_api_request_with_retry($api_url, array(
            'method' => 'POST',
            'timeout' => 180,
            'httpversion' => '1.0',
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-API-Key' => $bl_user_key,                    // BL User Key för autentisering
                'X-Organization-Number' => $organization_number, // Organisationsnummer för identifiering
                'X-WC-Consumer-Key' => $wc_consumer_key        // WooCommerce Consumer Key för validering
            ),
            'body' => json_encode($api_data)
        ), 'POST', true);
        
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            $this->log('Kunde inte starta integrationen efter retry-försök: ' . $error_message, 'error');
            wp_send_json_error('Kunde inte starta integrationen: ' . $error_message);
            return;
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if ($status_code === 200 && isset($body['status']) && $body['status'] === 'success') {
            wp_send_json_success($body);
        } else {
            $error_msg = isset($body['message']) ? $body['message'] : 'Okänt fel';
            wp_send_json_error('API-fel: ' . $error_msg);
        }
    }
    
    /**
     * Registrera webhook-endpoint för att ta emot anrop från Bjorn Lunden
     */
    public function register_webhook_endpoint() {
        add_rewrite_rule(
            'woo2bl-webhook/([^/]+)/([^/]+)/([^/]+)/?$',
            'index.php?woo2bl_public_key=$matches[1]&woo2bl_action=$matches[2]&woo2bl_org_nr=$matches[3]',
            'top'
        );
        
        add_filter('query_vars', array($this, 'add_webhook_query_vars'));
        add_action('parse_request', array($this, 'handle_webhook_request'));
        
        // Uppdatera rewrite-reglerna endast vid aktivering (hanteras i activate-metoden)
        // Inte här för att undvika att köra vid varje sidladdning
    }
    
    /**
     * Lägger till frågevariabler för webhook-anrop
     *
     * @param array $vars Nuvarande frågevariabler
     * @return array Uppdaterade frågevariabler
     */
    public function add_webhook_query_vars($vars) {
        $vars[] = 'woo2bl_public_key';
        $vars[] = 'woo2bl_action';
        $vars[] = 'woo2bl_org_nr';
        return $vars;
    }
    
    /**
     * AJAX-hanterare för webhook-förfrågningar från BL via admin-ajax.php
     */
    public function handle_webhook_request($wp) {
        // Kontrollera om det är en Bokför Ehandel webhook-begäran
        if (!isset($wp->query_vars['bokforehandel_webhook'])) {
            return;
        }

        // Rate limiting för webhook-endpoint
        $ip_address = $this->get_client_ip();
        $rate_limit_key = 'bokfore_webhook_rate_' . md5($ip_address);
        $requests = get_transient($rate_limit_key);
        
        if ($requests === false) {
            set_transient($rate_limit_key, 1, 60); // 1 minut
        } elseif ($requests >= 10) { // Max 10 requests per minut per IP
            error_log('Bokfor Ehandel: Webhook rate limit överskriden från IP: ' . $ip_address);
            status_header(429);
            die('Too many requests');
        } else {
            set_transient($rate_limit_key, $requests + 1, 60);
        }

        // Om vi är här, är det en webhook-begäran
        $data = file_get_contents('php://input');
        $data = json_decode($data, true);

        if (empty($data)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('Bokfor Ehandel: Mottog webhook utan data');
            }
            status_header(400);
            die('No data received');
        }

        // Verifiera att nödvändiga fält finns
        if (!isset($data['action']) || !isset($data['secret'])) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('Bokfor Ehandel: Webhook saknar nödvändiga fält: action eller secret');
            }
            status_header(400);
            die('Missing required fields');
        }

        // Extrahera data
        $action = sanitize_text_field($data['action']);
        $secret = sanitize_text_field($data['secret']);
        $org_nr = isset($data['org_nr']) ? sanitize_text_field($data['org_nr']) : '';
        
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log("Bokfor Ehandel: Webhook mottagen - action: $action, org_nr: $org_nr");
        }

        // Verifiera secret mot vårt API_SECRET
        $api_secret = get_option('bokforehandel_api_secret', '');
        // Fallback till environment variable om option inte finns
        if (empty($api_secret)) {
            $api_secret = getenv('BL_PUBLIC_KEY') ?: '';
        }
        
        if (empty($api_secret) || !hash_equals($api_secret, $secret)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('Bokfor Ehandel: Ogiltig webhook secret från IP: ' . $ip_address);
            }
            status_header(403);
            die('Invalid secret');
        }
        
        // Hantera olika åtgärder
        switch ($action) {
            case 'on':
                // Aktivera integrationen
                update_option('bokforehandel_integration_status', 'on');
                $this->log("Integration aktiverad för organisationsnummer $org_nr", 'info');
                
                // Anropa API för att meddela aktivering med fakturauppgifter
                $api_url = BOKFOREHANDEL_API_URL . '/api/webhook_received';
                
                // Samla alla fakturauppgifter
                $billing_data = array(
                    'billing_email' => get_option('bokforehandel_billing_email', ''),
                    'billing_reference' => get_option('bokforehandel_billing_reference', ''),
                    'billing_address' => get_option('bokforehandel_billing_address', ''),
                    'billing_plan' => get_option('bokforehandel_billing_price_option', 'monthly'),
                    'company_name' => get_option('bokforehandel_company_name', ''),
                    'organization_number' => get_option('bokforehandel_organization_number', ''),
                    'store_name' => get_option('bokforehandel_store_name', '')
                );
                
                // ÄNDRAT: Lägg till X-API-Key header för autentisering
                // API-servern kräver @require_api_key som validerar mot BLApiKey-tabellen i databasen
                // bokforehandel_bl_user_key är den BL User Key som användaren fyllt i och som sparas i databasen
                // Detta är säkert eftersom webhooks kommer från WordPress-pluginet (internt), inte externt
                $bl_user_key = get_option('bokforehandel_bl_user_key', '');
                if (empty($bl_user_key)) {
                    $this->log('Kan inte skicka webhook - BL User Key saknas', 'error');
                    http_response_code(500);
                    echo json_encode(array('success' => false, 'message' => 'BL User Key saknas'));
                    wp_die();
                }
                
                $response = wp_remote_post($api_url, array(
                    'method' => 'POST',
                    'timeout' => 10,
                    'httpversion' => '1.0',
                    'headers' => array(
                        'Content-Type' => 'application/json',
                        'X-API-Key' => $bl_user_key  // BL User Key för autentisering mot API-servern
                    ),
                    'body' => json_encode(array(
                        'action' => $action,
                        'org_nr' => $org_nr,
                        'secret' => $secret,
                        'billing_data' => $billing_data
                    ))
                ));
                
                if (is_wp_error($response)) {
                    error_log('Bokfor Ehandel: Fel vid anrop till API: ' . $response->get_error_message());
                    http_response_code(500);
                    echo json_encode(array('success' => false, 'message' => 'Fel vid anrop till API'));
                    wp_die();
                }
                
                http_response_code(200);
                echo json_encode(array('success' => true, 'message' => 'Integration aktiverad'));
                break;
                
            case 'off':
                // Stäng av integrationen
                update_option('bokforehandel_integration_status', 'off');
                $this->log("Integration inaktiverad för organisationsnummer $org_nr", 'info');
                
                // Anropa API för att meddela inaktivering
                $api_url = BOKFOREHANDEL_API_URL . '/api/webhook_received';
                
                // ÄNDRAT: Lägg till X-API-Key header för autentisering
                // API-servern kräver @require_api_key som validerar mot BLApiKey-tabellen i databasen
                // bokforehandel_bl_user_key är den BL User Key som användaren fyllt i och som sparas i databasen
                // Detta är säkert eftersom webhooks kommer från WordPress-pluginet (internt), inte externt
                $bl_user_key = get_option('bokforehandel_bl_user_key', '');
                if (empty($bl_user_key)) {
                    $this->log('Kan inte skicka webhook - BL User Key saknas', 'error');
                    http_response_code(500);
                    echo json_encode(array('success' => false, 'message' => 'BL User Key saknas'));
                    wp_die();
                }
                
                $response = wp_remote_post($api_url, array(
                    'method' => 'POST',
                    'timeout' => 10,
                    'httpversion' => '1.0',
                    'headers' => array(
                        'Content-Type' => 'application/json',
                        'X-API-Key' => $bl_user_key  // BL User Key för autentisering mot API-servern
                    ),
                    'body' => json_encode(array(
                        'action' => $action,
                        'org_nr' => $org_nr,
                        'secret' => $secret
                    ))
                ));
                
                http_response_code(200);
                echo json_encode(array('success' => true, 'message' => 'Integration inaktiverad'));
                break;
                
            default:
                $this->log("Okänd webhook-åtgärd: $action", 'warning');
                status_header(400);
                die('Unknown action');
        }
        
        wp_die(); // Viktigt: Stoppa WordPress från att fortsätta processa
    }

    /**
     * Skickar webhook till woo2bl API för att uppdatera integration status
     *
     * @param string $status 'on' eller 'off'
     * @return void
     */
    private function send_integration_webhook($status) {
        // Hämta nödvändig information
        $organization_number = get_option('bokforehandel_organization_number', '');
        $company_name = get_option('bokforehandel_company_name', '');
        $store_name = get_option('bokforehandel_store_name', '');

        if (empty($organization_number)) {
            error_log('Bokfor Ehandel: Kan inte skicka webhook - organisationsnummer saknas');
            return;
        }

        // Skapa API-URL
        $api_url = BOKFOREHANDEL_API_URL . '/api/webhook_received';

        // Förbered billing data för konsistens
        $billing_data = array(
            'billing_email' => get_option('bokforehandel_billing_email', ''),
            'billing_reference' => get_option('bokforehandel_billing_reference', ''),
            'billing_address' => get_option('bokforehandel_billing_address', ''),
            'billing_plan' => get_option('bokforehandel_billing_price_option', 'monthly'),
            'company_name' => $company_name,
            'organization_number' => $organization_number,
            'store_name' => $store_name
        );

        // Förbered webhook payload
        // Hämta API secret från WordPress options (eller environment variable som fallback)
        $api_secret = get_option('bokforehandel_api_secret', '');
        if (empty($api_secret)) {
            // Fallback till environment variable om option inte finns
            $api_secret = getenv('BL_PUBLIC_KEY') ?: 'default_secret';
        }
        
        $webhook_data = array(
            'action' => $status,
            'org_nr' => $organization_number,
            'secret' => $api_secret,
            'billing_data' => $billing_data
        );

        // ÄNDRAT: Lägg till X-API-Key header för autentisering
        // API-servern kräver @require_api_key som validerar mot BLApiKey-tabellen i databasen
        // bokforehandel_bl_user_key är den BL User Key som användaren fyllt i och som sparas i databasen
        // Detta är säkert eftersom webhooks kommer från WordPress-pluginet (internt), inte externt
        $bl_user_key = get_option('bokforehandel_bl_user_key', '');
        if (empty($bl_user_key)) {
            $this->log('Kan inte skicka webhook - BL User Key saknas', 'error');
            return; // Avbryt utan att krascha
        }
        
        // Skicka webhook
        // ÄNDRAT: Använd retry-logik för webhook-anrop (mindre kritisk, men viktig)
        $response = $this->make_api_request_with_retry($api_url, array(
            'method' => 'POST',
            'timeout' => 10,
            'httpversion' => '1.0',
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-API-Key' => $bl_user_key  // BL User Key för autentisering mot API-servern
            ),
            'body' => json_encode($webhook_data)
        ), 'POST', false); // Mindre kritisk - kortare retry-intervall

        // Logga resultat
        if (is_wp_error($response)) {
            error_log('Bokfor Ehandel: Fel vid webhook-anrop för integration status ' . $status . ': ' . $response->get_error_message());
        } else {
            $response_code = wp_remote_retrieve_response_code($response);
            $response_body = wp_remote_retrieve_body($response);
            error_log("Bokfor Ehandel: Webhook för integration status '$status' skickad. Status: $response_code, Response: $response_body");
        }
    }

    /**
     * Utför daglig synkronisering av ordrar från föregående dygn
     * Anropar Python-API:et för att starta synkroniseringsprocessen
     * 
     * ÄNDRAT: Uppdaterad med retry-logik och hantering av missade synkroniseringar
     * - Synkar först alla missade dagar (om några finns)
     * - Synkar sedan normal daglig synk (igår)
     * - Om synk misslyckas efter alla retry-försök, markeras datumet som missat
     */
    public function daily_sync_orders() {
        $this->log('Startar daglig synkronisering av föregående dags ordrar', 'info');
        
        // Kontrollera om integrations-ID och BL User Key finns
        $user_id = get_option('bokforehandel_user_id', '');
        $bl_user_key = get_option('bokforehandel_bl_user_key', '');
        
        if (empty($user_id) || empty($bl_user_key)) {
            $this->log('Daglig synkronisering avbruten - saknar användar-ID eller BL User Key', 'error');
            return;
        }
        
        // Kontrollera BL-integrationsstatus innan synkronisering
        $integration_status = $this->check_bl_integration_status($user_id);
        if ($integration_status !== 'on') {
            $this->log('Daglig synkronisering avbruten - integration är inte aktiv', 'info');
            return;
        }
        
        // Hämta nödvändiga värden för autentisering
        $organization_number = get_option('bokforehandel_organization_number', '');
        $wc_consumer_key = get_option('bokforehandel_wc_consumer_key', '');
        
        // Validera att alla nödvändiga värden finns
        if (empty($bl_user_key) || empty($organization_number) || empty($wc_consumer_key)) {
            $this->log('Daglig synkronisering avbruten - saknar nödvändiga API-nycklar eller organisationsnummer', 'error');
            return;
        }
        
        // ÄNDRAT: Först synka alla missade dagar (om några finns)
        $this->sync_missed_dates();
        
        // Konvertera datum till format som API-servern förväntar sig (YYYYMMDD-YYYYMMDD)
        // ÄNDRAT: Använd igår till idag (inte igår till igår) för att synka hela föregående dygn
        $yesterday = date('Ymd', strtotime('-1 day'));
        $today = date('Ymd');
        $date_range = $yesterday . '-' . $today;
        
        // Använd sync_date_range() som redan har retry-logik inbyggd
        $yesterday_date = date('Y-m-d', strtotime('-1 day'));
        $success = $this->sync_date_range($date_range, 'scheduled');
        
        if ($success) {
            // Spara senaste körningen
            $body = null;
            // Försök hämta run_id från senaste API-anropet (detta är en begränsning - vi kan inte enkelt få run_id här)
            // Men vi kan spara datumet ändå
            update_option('bokforehandel_last_sync', array(
                'date' => date('Y-m-d H:i:s'),
                'start_date' => $yesterday_date,
                'end_date' => date('Y-m-d'),
                'run_id' => 'scheduled'
            ));
            $this->log("Daglig synkronisering startad för datumintervall {$date_range}", 'info');
        } else {
            // ÄNDRAT: Om synk misslyckas efter alla retry-försök, markera datumet som missat
            $this->log("Daglig synkronisering misslyckades efter alla retry-försök för datum {$yesterday_date}", 'error');
            $this->mark_sync_date_as_missed($yesterday_date);
        }
    }
    
    /**
     * Hantera manuell synkronisering via AJAX
     * ÄNDRAT: Uppdaterad för att använda rätt API-endpoint och autentisering
     */
    public function ajax_manual_sync() {
        // Kontrollera nonce och behörigheter
        if (!$this->verify_ajax_security('bokforehandel_nonce')) {
            return;
        }
		
		// ÄNDRAT: Hämta rätt värden från inställningar
		// Använd BL User Key, Organization Number och WC Consumer Key istället för gamla api_key/user_id
		$bl_user_key = get_option('bokforehandel_bl_user_key', '');
		$organization_number = get_option('bokforehandel_organization_number', '');
		$wc_consumer_key = get_option('bokforehandel_wc_consumer_key', '');
		$user_id = get_option('bokforehandel_user_id', '');
		
		// Kontrollera om anslutningen är konfigurerad
		if (empty($bl_user_key) || empty($organization_number) || empty($wc_consumer_key) || empty($user_id)) {
			wp_send_json_error(array('message' => 'API-nycklar eller organisationsnummer saknas. Kontrollera dina inställningar.'));
		}
		
		// Kontrollera integrationsstatus
		$integration_status = get_option('bokforehandel_integration_status', 'not_configured');
		if ($integration_status === 'not_configured') {
			wp_send_json_error(array('message' => 'Integrationen är inte konfigurerad. Konfigurera integrationen först.'));
		}
		
		// Förbered operation_data för att köra synkronisering för igår
		$yesterday = date('Ymd', strtotime('yesterday'));
		$date_range = $yesterday . '-' . $yesterday;
		
		// Kontrollera om det redan finns en pågående eller nyligen slutförd körning för samma date_range och operation
		$existing_run = $this->check_existing_runs($date_range, 'all', $user_id);
		if ($existing_run) {
			$status_text = $existing_run['status'] === 'running' ? 'pågår redan' : 'slutfördes nyligen';
			$this->log("Förhindrar dubbelsynkning: En körning med samma date_range och operation {$status_text} (Körnings-ID: {$existing_run['id']})", 'info');
			wp_send_json_error(array('message' => "En synkronisering med samma datumintervall och operation {$status_text}. Vänta tills den är klar eller välj ett annat datumintervall."));
			wp_die();
		}
		
		// ÄNDRAT: Använd /api/run-integration istället för /api/run (som inte finns)
		
		// ÄNDRAT: Använd rätt dataformat för API-servern
		// API-servern förväntar sig 'operation' och 'date_range', inte 'operation_type' och separata datum
		$operation_data = array(
			'operation' => 'all',           // Använd 'all' för att synka både ordrar och utbetalningar
			'date_range' => $date_range,    // Format: YYYYMMDD-YYYYMMDD
			'operations_data' => array(
				'trigger' => 'manual',      // Flagga för att indikera att detta är manuell körning
				'is_daily_sync' => false
			)
		);
		
		// Logga API-anropet för felsökning
		$this->log('Manuell synkronisering API-anrop: ' . $api_url, 'info');
		$this->log('Manuell synkronisering data: ' . json_encode($operation_data), 'info');
		
		// ÄNDRAT: Använd retry-logik för kritiskt anrop
		// Tidigare: bara X-API-Key och X-User-ID (fel endpoint och fel autentisering)
		// Nu: X-API-Key, X-Organization-Number, X-WC-Consumer-Key (korrekt enligt API-serverns krav)
		// ÖKAD TIMEOUT: 180 sekunder (3 minuter) för att förhindra AJAX-fel vid hög serverbelastning
		$response = $this->make_api_request_with_retry($api_url, array(
			'method' => 'POST',
			'timeout' => 180,
			'httpversion' => '1.0',
			'headers' => array(
				'Content-Type' => 'application/json',
				'X-API-Key' => $bl_user_key,                    // BL User Key för autentisering
				'X-Organization-Number' => $organization_number, // Organisationsnummer för identifiering
				'X-WC-Consumer-Key' => $wc_consumer_key        // WooCommerce Consumer Key för validering
			),
			'body' => json_encode($operation_data)
		), 'POST', true);
		
		// Kontrollera om det är ett WP-fel
		if (is_wp_error($response)) {
			$error_message = $response->get_error_message();
			$this->log('Fel vid manuell synkronisering efter retry-försök: ' . $error_message, 'error');
			wp_send_json_error(array('message' => 'API-anropsfel: ' . $error_message));
		}
		
		// Få statuskod och svarstext
		$status_code = wp_remote_retrieve_response_code($response);
		$response_body = wp_remote_retrieve_body($response);
		$response_data = json_decode($response_body, true);
		
		// Logga API-svaret för felsökning
		$this->log('Manuell synkronisering API-svar: ' . $response_body, 'info');
		
		// ÄNDRAT: API-servern returnerar 'status' === 'success', inte 'success' === true
		// Kontrollera om förfrågan lyckades
		if ($status_code == 200 && isset($response_data['status']) && $response_data['status'] === 'success') {
			// Uppdatera senaste synkroniseringsdatum
			update_option('bokforehandel_last_sync_date', current_time('mysql'));
			
			wp_send_json_success(array(
				'message' => 'Manuell synkronisering har startats. Bearbetar data...',
				'data' => $response_data
			));
		} else {
			// Det blev ett fel från API:et
			$error_message = isset($response_data['message']) ? $response_data['message'] : 'Okänt fel från API';
			$this->log('Manuell synkronisering misslyckades: ' . $error_message, 'error');
			wp_send_json_error(array(
				'message' => 'API-fel: ' . $error_message,
				'status_code' => $status_code
			));
		}
		
		wp_die();
    }
    
    /**
     * AJAX: Hämta synkroniserade ordrar från en körning
     */
    public function ajax_get_synced_orders() {
        // Kontrollera nonce och behörighet
        if (!$this->verify_ajax_security('bokforehandel-get-synced-orders', false)) {
            $this->log('Nonce-verifiering misslyckades vid hämtning av synkroniserade ordrar', 'error');
            wp_send_json_error(['message' => 'Säkerhetsverifiering misslyckades']);
            return;
        }
        
        $run_id = isset($_POST['run_id']) ? intval($_POST['run_id']) : 0;
        if ($run_id <= 0) {
            $this->log('Ogiltigt körnings-ID vid hämtning av synkroniserade ordrar: ' . $run_id, 'error');
            wp_send_json_error(['message' => 'Ogiltigt körnings-ID']);
            return;
        }
        
        $user_id = get_option('bokforehandel_user_id', '');
        if (empty($user_id)) {
            $this->log('Användar-ID saknas vid hämtning av synkroniserade ordrar', 'error');
            wp_send_json_error(['message' => 'Användar-ID saknas. Kontrollera inställningarna.']);
            return;
        }
        
        // Logga för felsökning
        error_log('Bokforehandel: Hämtar synkroniserade ordrar för körning ' . $run_id . ' och användare ' . $user_id);
        
        // Gör API-anrop till korrekt endpoint med API-nyckel
        $api_url = BOKFOREHANDEL_API_URL . '/api/run/' . $run_id . '/orders';
        error_log('Bokforehandel: API URL för synkroniserade ordrar: ' . $api_url);
        
        $response = wp_remote_get($api_url, [
            'timeout' => 10,
            'headers' => [
                'Content-Type' => 'application/json',
                'X-API-Key' => get_option('bokforehandel_bl_user_key'),
                'X-User-ID' => $user_id
            ]
        ]);
        
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            $this->log('API-fel vid hämtning av synkroniserade ordrar: ' . $error_message, 'error');
            wp_send_json_error(['message' => 'Anslutningsfel: ' . $error_message]);
            return;
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        error_log('Bokforehandel: API-svar statuskod vid hämtning av synkroniserade ordrar: ' . $status_code);
        error_log('Bokforehandel: API-svar body vid hämtning av synkroniserade ordrar: ' . $response_body);
        
        $body = json_decode($response_body, true);
        
        if ($status_code === 200 && isset($body['status']) && $body['status'] === 'success') {
            if (isset($body['data']) && is_array($body['data'])) {
                wp_send_json_success(['orders' => $body['data']]);
            } else {
                // API returnerade OK men ingen data
                wp_send_json_success(['orders' => []]);
            }
        } else {
            $error_msg = isset($body['message']) ? $body['message'] : 'Okänt fel';
            error_log('Bokforehandel: API-fel vid hämtning av synkroniserade ordrar: ' . $error_msg);
            wp_send_json_error(['message' => 'API-fel: ' . $error_msg]);
        }
    }

    /**
     * Hämta alla tillgängliga betalningsmetoder från WooCommerce
     * 
     * @return array Lista med betalningsmetoder (ID => namn)
     */
    public function get_payment_gateways() {
        $gateways = array();
        
        // Hämta WooCommerce-gateways om tillgängliga
        if (function_exists('WC') && isset(WC()->payment_gateways) && is_object(WC()->payment_gateways)) {
            $available_gateways = WC()->payment_gateways->get_available_payment_gateways();
            
            if (is_array($available_gateways)) {
                foreach ($available_gateways as $gateway_id => $gateway) {
                    $gateways[$gateway_id] = $gateway->get_title();
                }
            }
        }
        
        // Lägg till standardmetoder om inga hittas eller som fallback
        if (empty($gateways)) {
            $gateways = array(
                'kco' => 'Klarna Checkout',
                'stripe' => 'Stripe',
                'swish' => 'Swish',
                'other' => 'Övriga betalningsmetoder'
            );
        } else {
            // Lägg alltid till "Övriga" som ett alternativ
            $gateways['other'] = 'Övriga betalningsmetoder';
        }
        
        return $gateways;
    }

    /**
     * Hämta API-URL baserat på miljö
     * 
     * @return string API-URL
     */
    private function get_api_url() {
        // Använd den fördefinierade konstanten för API-URL
        $api_url = defined('BOKFOREHANDEL_API_URL') ? BOKFOREHANDEL_API_URL : 'https://woo2bl.ideang.se';

        // Trimma bort eventuellt avslutande snedstreck
        return rtrim($api_url, '/');
    }

    /**
     * Extrahera store_url från wc_api_url
     *
     * @param string $wc_api_url URL till WooCommerce API (t.ex. https://example.com/wp-json/wc/v3)
     * @return string Store URL (t.ex. https://example.com)
     */
    private function extract_store_url($wc_api_url) {
        // Ta bort /wp-json/wc/v3 och allt efter det
        $url_parts = parse_url($wc_api_url);
        if (isset($url_parts['scheme']) && isset($url_parts['host'])) {
            $store_url = $url_parts['scheme'] . '://' . $url_parts['host'];
            // Lägg till port om den finns
            if (isset($url_parts['port']) && $url_parts['port'] != 80 && $url_parts['port'] != 443) {
                $store_url .= ':' . $url_parts['port'];
            }
            return $store_url;
        }
        return $wc_api_url;
    }

    /**
     * Hämta butikens valuta från WooCommerce
     *
     * @return string ISO 4217 valutakod (t.ex. 'EUR', 'SEK', 'USD')
     */
    private function get_store_currency() {
        // Först försök hämta från WooCommerce-inställningar
        if (function_exists('get_woocommerce_currency')) {
            $currency = get_woocommerce_currency();
            if (!empty($currency)) {
                error_log('Bokforehandel: Valuta detekterad från WooCommerce: ' . $currency);
                return strtoupper($currency);
            }
        }
        
        // Fallback: Hämta från senaste order
        if (class_exists('WC_Order')) {
            $args = array(
                'limit' => 1,
                'orderby' => 'date',
                'order' => 'DESC',
                'return' => 'objects'
            );
            
            $orders = wc_get_orders($args);
            if (!empty($orders) && isset($orders[0])) {
                $order_currency = $orders[0]->get_currency();
                if (!empty($order_currency)) {
                    error_log('Bokforehandel: Valuta detekterad från senaste order: ' . $order_currency);
                    return strtoupper($order_currency);
                }
            }
        }
        
        // Sista fallback: SEK
        error_log('Bokforehandel: Kunde inte detektera valuta, använder SEK som standard');
        return 'SEK';
    }

    /**
     * Kontrollera efter plugin-uppdateringar från vår server
     * 
     * @param object $transient WordPress uppdateringstransient
     * @return object Modified transient
     */
    public function plugin_update_check($transient) {
        if (empty($transient->checked)) {
            return $transient;
        }
        
        $plugin_slug = plugin_basename(BOKFOREHANDEL_PLUGIN_FILE);
        
        // Hämta uppdateringsinformation från vår server
        $remote_info = $this->get_remote_plugin_info();
        
        if ($remote_info && version_compare(BOKFOREHANDEL_VERSION, $remote_info->version, '<')) {
            $plugin_update = array(
                'slug' => 'bokfor-ehandel',
                'new_version' => $remote_info->version,
                'url' => $remote_info->homepage,
                'package' => $remote_info->download_url,
                'tested' => $remote_info->tested,
                'requires_php' => $remote_info->requires_php,
                'compatibility' => new stdClass()
            );
            
            $transient->response[$plugin_slug] = (object) $plugin_update;
        }
        
        return $transient;
    }
    
    /**
     * Tillhandahåll plugin-information för uppdateringsvisning
     * 
     * @param false|object|array $result
     * @param string $action
     * @param object $args
     * @return false|object
     */
    public function plugin_update_info($result, $action, $args) {
        if ($action !== 'plugin_information') {
            return $result;
        }
        
        if ($args->slug !== 'bokfor-ehandel') {
            return $result;
        }
        
        $remote_info = $this->get_remote_plugin_info();
        
        if (!$remote_info) {
            return $result;
        }
        
        return $remote_info;
    }
    
    /**
     * Hämta plugin-information från vår uppdateringsserver
     * 
     * @return object|false Plugin-information eller false vid fel
     */
    private function get_remote_plugin_info() {
        // Cacha resultatet i 12 timmar
        $cache_key = 'bokforehandel_update_info';
        $cached = get_transient($cache_key);
        
        if ($cached !== false) {
            return $cached;
        }
        
        $update_url = BOKFOREHANDEL_API_URL . '/api/plugin-update-info';
        
        $response = wp_remote_get($update_url, array(
            'timeout' => 10,
            'headers' => array(
                'Accept' => 'application/json'
            )
        ));
        
        if (is_wp_error($response)) {
            error_log('Bokför Ehandel: Kunde inte hämta uppdateringsinformation: ' . $response->get_error_message());
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body);
        
        if (!$data || !isset($data->version)) {
            error_log('Bokför Ehandel: Ogiltig uppdateringsinformation mottagen');
            return false;
        }
        
        // Cacha i 12 timmar
        set_transient($cache_key, $data, 12 * HOUR_IN_SECONDS);
        
        return $data;
    }

    /**
     * Hämta klientens IP-adress
     * 
     * @return string IP-adress
     */
    private function get_client_ip() {
        $ip_keys = array(
            'HTTP_CF_CONNECTING_IP', // Cloudflare
            'HTTP_X_REAL_IP',
            'HTTP_X_FORWARDED_FOR',
            'REMOTE_ADDR'
        );
        
        foreach ($ip_keys as $key) {
            if (array_key_exists($key, $_SERVER) === true) {
                $ip = sanitize_text_field($_SERVER[$key]);
                // Om X-Forwarded-For kan innehålla flera IPs, ta den första
                if (strpos($ip, ',') !== false) {
                    $ip = explode(',', $ip)[0];
                }
                // Validera IP
                if (filter_var(trim($ip), FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                    return trim($ip);
                }
            }
        }
        
        // Fallback till REMOTE_ADDR
        return isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field($_SERVER['REMOTE_ADDR']) : '0.0.0.0';
    }
    
    /**
     * Logga meddelanden för felsökning
     * Använder WordPress logger om tillgänglig, annars error_log endast i debug-läge
     * 
     * @param string $message Meddelande att logga
     * @param string $level Loggnivå (info, error, warning)
     * @return void
     */
    public function log($message, $level = 'info') {
        // Logga endast om WP_DEBUG är aktiverat eller om det är ett fel
        if (!defined('WP_DEBUG') || !WP_DEBUG) {
            if ($level !== 'error') {
                return; // Logga endast fel i produktion
            }
        }
        
        // Försök använda WooCommerce logger om tillgänglig
        if (function_exists('wc_get_logger') && $level === 'error') {
            $logger = wc_get_logger();
            $logger->error($message, array('source' => 'bokforehandel'));
            return;
        }
        
        $prefix = '[Bokför Ehandel]';
        
        switch ($level) {
            case 'error':
                $prefix .= ' [ERROR]';
                break;
            case 'warning':
                $prefix .= ' [WARNING]';
                break;
            default:
                $prefix .= ' [INFO]';
                break;
        }
        
        error_log($prefix . ' ' . $message);
    }
    
    /**
     * Gör API-anrop med retry-logik för att hantera tillfälliga fel och långa avbrott
     * 
     * @param string $url API-URL
     * @param array $args Argument för wp_remote_post/wp_remote_get
     * @param string $method HTTP-metod ('POST' eller 'GET')
     * @param bool $is_critical Om detta är ett kritiskt anrop (använder längre retry-intervall)
     * @return array|WP_Error Response från API-anropet
     */
    private function make_api_request_with_retry($url, $args, $method = 'POST', $is_critical = true) {
        // För kritiska anrop: 9 försök över 24h, för mindre kritiska: 3 försök över 1min
        $max_attempts = $is_critical ? 9 : 3;
        $intervals = $is_critical 
            ? array(1, 5, 15, 60, 300, 900, 3600, 21600, 86400) // 1s, 5s, 15s, 1m, 5m, 15m, 1h, 6h, 24h
            : array(1, 5, 15); // 1s, 5s, 15s
        
        $attempt = 0;
        $last_error = null;
        $last_response = null;
        
        while ($attempt < $max_attempts) {
            $attempt++;
            
            $this->log("API-anrop försök {$attempt}/{$max_attempts}: {$method} {$url}", 'info');
            
            // Gör API-anrop
            if ($method === 'POST') {
                $response = wp_remote_post($url, $args);
            } else {
                $response = wp_remote_get($url, $args);
            }
            
            $last_response = $response;
            
            // Kontrollera om retry behövs
            $should_retry = $this->should_retry_request($response, $attempt, $max_attempts);
            
            if (!$should_retry) {
                if ($attempt > 1) {
                    $this->log("API-anrop lyckades efter {$attempt} försök", 'info');
                }
                return $response; // Framgång eller permanent fel
            }
            
            // Spara fel för loggning
            if (is_wp_error($response)) {
                $last_error = $response->get_error_message();
                $error_code = $response->get_error_code();
                $this->log("API-anrop misslyckades (WP_Error): {$error_code} - {$last_error}", 'warning');
            } else {
                $status_code = wp_remote_retrieve_response_code($response);
                $last_error = "HTTP {$status_code}";
                $this->log("API-anrop misslyckades med statuskod: {$status_code}", 'warning');
            }
            
            // Vänta innan nästa försök (exponential backoff)
            if ($attempt < $max_attempts) {
                $delay = $intervals[$attempt - 1];
                $delay_human = $this->format_delay($delay);
                $this->log("API-anrop misslyckades ({$last_error}), försöker igen om {$delay_human} (försök {$attempt}/{$max_attempts})", 'warning');
                
                // För längre delays (> 60 sekunder), använd WordPress cron istället för sleep
                // för att undvika att blockera PHP-processen
                // SÄKERHET: Ta bort API-nycklar från args innan de sparas i cron-databasen
                if ($delay > 60) {
                    // Schemalägg nästa försök via WordPress cron
                    $hook_name = 'bokforehandel_retry_api_call';
                    
                    // SÄKERHET: Skapa en säker kopia av args utan API-nycklar i headers
                    // API-nycklarna hämtas från options när retry körs istället
                    $safe_args = $args;
                    if (isset($safe_args['headers'])) {
                        // Ta bort känsliga headers - de hämtas från options vid retry
                        unset($safe_args['headers']['X-API-Key']);
                        unset($safe_args['headers']['X-WC-Consumer-Key']);
                        // Behåll X-Organization-Number eftersom det inte är känsligt
                    }
                    
                    $args_for_cron = array($url, $safe_args, $method, $is_critical, $attempt + 1);
                    
                    // Registrera hook om den inte redan finns
                    if (!has_action($hook_name)) {
                        add_action($hook_name, array($this, 'handle_scheduled_retry'));
                    }
                    
                    wp_schedule_single_event(time() + $delay, $hook_name, $args_for_cron);
                    $this->log("Schemalagt retry-försök {$attempt} via WordPress cron om {$delay_human}", 'info');
                    return $response; // Returnera nuvarande fel, retry kommer via cron
                } else {
                    sleep($delay);
                }
            }
        }
        
        // Alla försök misslyckades
        $this->log("API-anrop misslyckades efter {$max_attempts} försök. Sista fel: {$last_error}", 'error');
        return $last_response; // Returnera sista försöket
    }
    
    /**
     * Hantera schemalagt retry-försök via WordPress cron
     * 
     * SÄKERHET: API-nycklar hämtas från WordPress options istället för att sparas i cron-argumenten
     * 
     * @param string $url API-URL
     * @param array $args Argument för API-anropet (utan API-nycklar i headers)
     * @param string $method HTTP-metod
     * @param bool $is_critical Om detta är kritiskt anrop
     * @param int $current_attempt Nuvarande försöksnummer
     */
    public function handle_scheduled_retry($url, $args, $method, $is_critical, $current_attempt) {
        $this->log("Kör schemalagt retry-försök {$current_attempt} för {$method} {$url}", 'info');
        
        // SÄKERHET: Återställ API-nycklar från WordPress options
        // Detta säkerställer att API-nycklar inte sparas i cron-databasen
        if (!isset($args['headers'])) {
            $args['headers'] = array();
        }
        
        // Hämta API-nycklar från options baserat på URL
        // Om det är /api/run-integration, behöver vi trippel-autentisering
        if (strpos($url, '/api/run-integration') !== false || strpos($url, '/api/webhook_received') !== false) {
            $bl_user_key = get_option('bokforehandel_bl_user_key', '');
            $organization_number = get_option('bokforehandel_organization_number', '');
            $wc_consumer_key = get_option('bokforehandel_wc_consumer_key', '');
            
            if (!empty($bl_user_key)) {
                $args['headers']['X-API-Key'] = $bl_user_key;
            }
            if (!empty($organization_number)) {
                $args['headers']['X-Organization-Number'] = $organization_number;
            }
            if (!empty($wc_consumer_key)) {
                $args['headers']['X-WC-Consumer-Key'] = $wc_consumer_key;
            }
        } else {
            // För andra endpoints, använd bara X-API-Key
            $bl_user_key = get_option('bokforehandel_bl_user_key', '');
            if (!empty($bl_user_key)) {
                $args['headers']['X-API-Key'] = $bl_user_key;
            }
        }
        
        // Säkerställ att Content-Type finns
        if (!isset($args['headers']['Content-Type'])) {
            $args['headers']['Content-Type'] = 'application/json';
        }
        
        // Fortsätt retry-logiken från där vi slutade
        $max_attempts = $is_critical ? 9 : 3;
        $intervals = $is_critical 
            ? array(1, 5, 15, 60, 300, 900, 3600, 21600, 86400)
            : array(1, 5, 15);
        
        // Gör API-anrop
        if ($method === 'POST') {
            $response = wp_remote_post($url, $args);
        } else {
            $response = wp_remote_get($url, $args);
        }
        
        $should_retry = $this->should_retry_request($response, $current_attempt, $max_attempts);
        
        if (!$should_retry) {
            $this->log("Schemalagt retry-försök {$current_attempt} lyckades", 'info');
            return $response;
        }
        
        // Om vi behöver fortsätta retry
        if ($current_attempt < $max_attempts) {
            $delay = isset($intervals[$current_attempt - 1]) ? $intervals[$current_attempt - 1] : $intervals[count($intervals) - 1];
            $delay_human = $this->format_delay($delay);
            
            $hook_name = 'bokforehandel_retry_api_call';
            
            // SÄKERHET: Ta bort API-nycklar igen innan nästa cron-schemaläggning
            $safe_args = $args;
            if (isset($safe_args['headers'])) {
                unset($safe_args['headers']['X-API-Key']);
                unset($safe_args['headers']['X-WC-Consumer-Key']);
            }
            
            $args_for_cron = array($url, $safe_args, $method, $is_critical, $current_attempt + 1);
            
            wp_schedule_single_event(time() + $delay, $hook_name, $args_for_cron);
            $this->log("Schemalagt nästa retry-försök {$current_attempt} om {$delay_human}", 'warning');
        } else {
            $this->log("Schemalagt retry misslyckades efter {$max_attempts} försök", 'error');
        }
        
        return $response;
    }
    
    /**
     * Avgör om ett API-anrop bör försökas igen
     * 
     * @param array|WP_Error $response Response från API-anropet
     * @param int $attempt Nuvarande försöksnummer
     * @param int $max_attempts Maximalt antal försök
     * @return bool Om retry bör göras
     */
    private function should_retry_request($response, $attempt, $max_attempts) {
        // Om vi har gjort alla försök, retry inte
        if ($attempt >= $max_attempts) {
            return false;
        }
        
        // Retry vid nätverksfel/timeout
        if (is_wp_error($response)) {
            $error_code = $response->get_error_code();
            // Retry vid timeout eller nätverksfel
            if (in_array($error_code, array('http_request_failed', 'timeout', 'connect_timeout'))) {
                return true;
            }
            return false; // Andra fel är permanenta
        }
        
        // Retry vid serverfel (5xx) eller rate limit (429)
        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code >= 500 && $status_code < 600) {
            return true; // Serverfel - försök igen
        }
        if ($status_code === 429) {
            return true; // Rate limit - försök igen med längre delay
        }
        
        // Inte retry vid klientfel (4xx) eller framgång (2xx)
        return false;
    }
    
    /**
     * Formatera delay-tid i läsbart format
     * 
     * @param int $seconds Antal sekunder
     * @return string Formaterad sträng (t.ex. "5s", "2min", "1h", "3d")
     */
    private function format_delay($seconds) {
        if ($seconds < 60) {
            return "{$seconds}s";
        }
        if ($seconds < 3600) {
            return round($seconds / 60) . "min";
        }
        if ($seconds < 86400) {
            return round($seconds / 3600) . "h";
        }
        return round($seconds / 86400) . "d";
    }
    
    /**
     * Markera ett datum som missat synkronisering
     * 
     * @param string|DateTime $date Datum att markera som missat (format 'Y-m-d' eller DateTime-objekt)
     */
    private function mark_sync_date_as_missed($date) {
        $missed_dates = get_option('bokforehandel_missed_sync_dates', array());
        $date_str = is_string($date) ? $date : $date->format('Y-m-d');
        
        if (!in_array($date_str, $missed_dates)) {
            $missed_dates[] = $date_str;
            sort($missed_dates); // Sortera för att synka äldsta först
            update_option('bokforehandel_missed_sync_dates', $missed_dates);
            $this->log("Markerat datum som missat: {$date_str}. Totalt missade dagar: " . count($missed_dates), 'warning');
        }
    }
    
    /**
     * Hämta lista över missade synkroniseringsdatum
     * 
     * @return array Array med datum i format 'Y-m-d'
     */
    private function get_missed_sync_dates() {
        return get_option('bokforehandel_missed_sync_dates', array());
    }
    
    /**
     * Ta bort ett datum från lista över missade synkroniseringar
     * 
     * @param string|DateTime $date Datum att ta bort (format 'Y-m-d' eller DateTime-objekt)
     */
    private function clear_missed_sync_date($date) {
        $missed_dates = get_option('bokforehandel_missed_sync_dates', array());
        $date_str = is_string($date) ? $date : $date->format('Y-m-d');
        
        $missed_dates = array_values(array_filter($missed_dates, function($d) use ($date_str) {
            return $d !== $date_str;
        }));
        
        update_option('bokforehandel_missed_sync_dates', $missed_dates);
        $this->log("Rensat missat datum: {$date_str}. Återstående missade dagar: " . count($missed_dates), 'info');
    }
    
    /**
     * Synka alla missade dagar sekventiellt
     * 
     * @return bool Om synkning av missade dagar påbörjades
     */
    private function sync_missed_dates() {
        $missed_dates = $this->get_missed_sync_dates();
        
        if (empty($missed_dates)) {
            return true; // Inga missade dagar
        }
        
        $this->log("Hittade " . count($missed_dates) . " missade dagar att synka: " . implode(', ', $missed_dates), 'warning');
        
        // Synka äldsta först (första i listan eftersom den är sorterad)
        foreach ($missed_dates as $missed_date) {
            $this->log("Synkar missad dag: {$missed_date}", 'info');
            
            // Konvertera datum till YYYYMMDD-YYYYMMDD format
            $date_obj = DateTime::createFromFormat('Y-m-d', $missed_date);
            if (!$date_obj) {
                $this->log("Ogiltigt datumformat för missad dag: {$missed_date}", 'error');
                $this->clear_missed_sync_date($missed_date); // Ta bort ogiltigt datum
                continue;
            }
            
            $date_range = $date_obj->format('Ymd') . '-' . $date_obj->format('Ymd');
            
            // Anropa API för att synka detta datum
            $success = $this->sync_date_range($date_range, "missed_sync_{$missed_date}");
            
            if ($success) {
                $this->clear_missed_sync_date($missed_date);
                $this->log("Missad dag synkad framgångsrikt: {$missed_date}", 'info');
            } else {
                $this->log("Misslyckades att synka missad dag: {$missed_date}. Kommer försöka igen vid nästa körning", 'error');
                break; // Stoppa här, försök igen nästa gång
            }
        }
        
        return true;
    }
    
    /**
     * Synka ett specifikt datumintervall
     * 
     * @param string $date_range Datumintervall i format YYYYMMDD-YYYYMMDD
     * @param string $trigger Typ av trigger (t.ex. 'missed_sync_2024-01-01' eller 'scheduled')
     * @return bool Om synkningen lyckades
     */
    private function sync_date_range($date_range, $trigger = 'scheduled') {
        // Hämta nödvändiga värden för autentisering
        $bl_user_key = get_option('bokforehandel_bl_user_key', '');
        $organization_number = get_option('bokforehandel_organization_number', '');
        $wc_consumer_key = get_option('bokforehandel_wc_consumer_key', '');
        
        if (empty($bl_user_key) || empty($organization_number) || empty($wc_consumer_key)) {
            $this->log('Kan inte synka - saknar nödvändiga API-nycklar eller organisationsnummer', 'error');
            return false;
        }
        
        $api_url = BOKFOREHANDEL_API_URL . '/api/run-integration';
        
        $api_data = array(
            'operation' => 'all',
            'date_range' => $date_range,
            'operations_data' => array(
                'trigger' => $trigger,
                'is_daily_sync' => ($trigger === 'scheduled')
            )
        );
        
        // Använd retry-logik för kritiskt anrop
        // ÖKAD TIMEOUT: 180 sekunder (3 minuter) för att förhindra AJAX-fel vid hög serverbelastning
        $response = $this->make_api_request_with_retry($api_url, array(
            'method' => 'POST',
            'timeout' => 180,
            'httpversion' => '1.0',
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-API-Key' => $bl_user_key,
                'X-Organization-Number' => $organization_number,
                'X-WC-Consumer-Key' => $wc_consumer_key
            ),
            'body' => json_encode($api_data)
        ), 'POST', true);
        
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            $this->log("Fel vid synkning av datumintervall {$date_range}: {$error_message}", 'error');
            return false;
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code >= 200 && $status_code < 300) {
            $body = json_decode(wp_remote_retrieve_body($response), true);
            $run_id = isset($body['run_id']) ? $body['run_id'] : 'okänt';
            $this->log("Synkning av datumintervall {$date_range} startad med run_id: {$run_id}", 'info');
            return true;
        } else {
            $this->log("Synkning av datumintervall {$date_range} misslyckades med status: {$status_code}", 'error');
            return false;
        }
    }
    
    /**
     * Hämta användar-ID från headers
     * 
     * @return string Användar-ID eller tom sträng
     */
    private function get_user_id_from_headers() {
        $headers = function_exists('getallheaders') ? getallheaders() : array();
        
        if (isset($headers['X-User-ID'])) {
            return sanitize_text_field($headers['X-User-ID']);
        }
        
        return '';
    }

    /**
     * Validerar organisationsnummer (svenskt format: 123456-7890 eller 1234567890)
     * 
     * @param string $org_nr Organisationsnummer att validera
     * @return bool Om organisationsnumret är giltigt
     */
    private function validate_organization_number($org_nr) {
        // Ta bort eventuella mellanslag och bindestreck
        $org_nr = preg_replace('/[\s\-]/', '', $org_nr);
        
        // Kontrollera att det är 10 siffror
        if (!preg_match('/^\d{10}$/', $org_nr)) {
            return false;
        }
        
        // Kontrollera kontrollsiffran (Luhn-algoritm)
        $sum = 0;
        for ($i = 0; $i < 9; $i++) {
            $digit = intval($org_nr[$i]);
            $multiplier = ($i % 2 === 0) ? 2 : 1;
            $product = $digit * $multiplier;
            $sum += ($product > 9) ? $product - 9 : $product;
        }
        
        $check_digit = (10 - ($sum % 10)) % 10;
        return $check_digit === intval($org_nr[9]);
    }
    
    /**
     * Validerar kontonummer (4 siffror)
     * 
     * @param string $account Kontonummer att validera
     * @return bool Om kontonumret är giltigt
     */
    private function validate_account_number($account) {
        // Ta bort eventuella mellanslag
        $account = trim($account);
        
        // Kontrollera att det är 4 siffror
        return preg_match('/^\d{4}$/', $account) === 1;
    }
    
    /**
     * Säkerställer att ett datum är i formatet YYYY-MM-DD
     */
    private function ensure_date_format($date) {
        // Kontrollera om datumet redan är i rätt format
        if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $date)) {
            return $date;
        }
        
        // Försök tolka datumet
        $timestamp = strtotime($date);
        if ($timestamp === false) {
            return $date; // Kan inte tolka, returnera originalet
        }
        
        return date('Y-m-d', $timestamp);
    }
    
    /**
     * Validerar att ett datum är i formatet YYYY-MM-DD
     */
    private function validate_date_format($date) {
        if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $date)) {
            return false;
        }
        
        $parts = explode('-', $date);
        return checkdate($parts[1], $parts[2], $parts[0]);
    }

    /**
     * AJAX: Update settings
     */
    public function ajax_update_settings() {
        // Check nonce and user permissions
        if ( !wp_verify_nonce( $_POST['security'], 'bokforehandel-ajax-nonce' ) || !current_user_can( 'manage_options' ) ) {
            wp_send_json_error( array(
                'message' => __( 'Security check failed.', 'bokforehandel' ),
            ) );
        }

        // Update settings
        $this->log( 'Updating settings with data: ' . print_r( $_POST, true ) );
        
        update_option( 'bokforehandel_api_url', sanitize_text_field( $_POST['api_url'] ) );
        update_option( 'bokforehandel_user_id', sanitize_text_field( $_POST['user_id'] ) );
        update_option( 'bokforehandel_api_key', sanitize_text_field( $_POST['api_key'] ) );
        update_option( 'bokforehandel_debug_mode', isset( $_POST['debug_mode'] ) ? 'on' : 'off' );
        
        // WooCommerce API settings
        update_option( 'bokforehandel_wc_api_enabled', isset( $_POST['wc_api_enabled'] ) ? 'on' : 'off' );
        update_option( 'bokforehandel_wc_api_consumer_key', sanitize_text_field( $_POST['wc_api_consumer_key'] ) );
        update_option( 'bokforehandel_wc_api_consumer_secret', sanitize_text_field( $_POST['wc_api_consumer_secret'] ) );
        
        $this->log( 'Settings updated successfully.' );
    }

    /**
     * AJAX: Update auto-sync settings only
     */
    public function ajax_update_auto_sync() {
        // Verifiera nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'bokforehandel_nonce')) {
            wp_send_json_error('Säkerhetsverifiering misslyckades');
        }
        
        // Kontrollera användarrättigheter
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Du har inte behörighet att ändra dessa inställningar');
        }
        
        // Uppdatera enbart auto-sync inställningen
        $sync_enabled = isset($_POST['sync_enabled']) ? sanitize_text_field($_POST['sync_enabled']) : 'no';
        update_option('bokforehandel_sync_enabled', $sync_enabled);
        
        $this->log('Auto-sync inställningen uppdaterad till: ' . $sync_enabled);
        
        wp_send_json_success('Autosynkronisering har uppdaterats');
    }

    public function enqueue_scripts() {
        wp_enqueue_script('bokforehandel-admin', plugin_dir_url(__FILE__) . 'assets/js/admin.js', ['jquery'], '1.0', true);
        wp_localize_script('bokforehandel-admin', 'apiParams', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('bokforehandel_nonce')
        ]);
    }

    /**
     * Företagsinställningssidan
     */
    public function company_settings_page() {
        // Ladda mallen direkt istället för via wp_loaded
        include(BOKFOREHANDEL_PLUGIN_DIR . 'templates/company_settings.php');
    }
    
    /**
     * Bokföringsinställningssidan
     */
    public function accounting_settings_page() {
        // Ladda mallen direkt istället för via wp_loaded
        include(BOKFOREHANDEL_PLUGIN_DIR . 'templates/accounting_settings.php');
    }
    
    /**
     * Synkroniseringsinställningssidan (inaktiverad - omdirigeras nu till körningshistorik)
     */
    public function sync_settings_page() {
        // Omdirigera till körningshistorik om javascript inte hanterar det
        echo '<script>window.location.href = "' . admin_url('admin.php?page=bokforehandel-runs') . '";</script>';
        echo '<p>Omdirigerar till <a href="' . admin_url('admin.php?page=bokforehandel-runs') . '">Körningshistorik</a>...</p>';
        exit;
    }

    /**
     * AJAX: Spara företagsinställningar
     */
    public function ajax_save_company() {
        // Kontrollera AJAX-nonce
        if (!$this->verify_ajax_security('bokforehandel_nonce')) {
            return;
        }
    
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Du har inte behörighet att utföra denna åtgärd');
            return;
        }
        
        // Sanera och spara företagsnamn
        if (isset($_POST['company_name']) && !empty($_POST['company_name'])) {
            update_option('bokforehandel_company_name', sanitize_text_field($_POST['company_name']));
        } else {
            wp_send_json_error('Företagsnamn är obligatoriskt');
            return;
        }
        
        // Sanera och spara butiksnamn
        if (isset($_POST['store_name']) && !empty($_POST['store_name'])) {
            update_option('bokforehandel_store_name', sanitize_text_field($_POST['store_name']));
        } else {
            wp_send_json_error('Butiksnamn är obligatoriskt');
            return;
        }
        
        // Validera och spara organisationsnummer
        if (isset($_POST['organization_number']) && !empty($_POST['organization_number'])) {
            $org_nr = sanitize_text_field($_POST['organization_number']);
            if (!$this->validate_organization_number($org_nr)) {
                wp_send_json_error('Ogiltigt organisationsnummer. Ange ett giltigt svenskt organisationsnummer (10 siffror, t.ex. 559123-1234).');
                return;
            }
            update_option('bokforehandel_organization_number', $org_nr);
        } else {
            wp_send_json_error('Organisationsnummer är obligatoriskt');
            return;
        }
        
        // Kontrollera och spara faktureringsinformation
        if (isset($_POST['billing_email']) && !empty($_POST['billing_email'])) {
            if (!is_email($_POST['billing_email'])) {
                wp_send_json_error('Ange en giltig e-postadress för fakturering');
                return;
            }
            update_option('bokforehandel_billing_email', sanitize_email($_POST['billing_email']));
        } else {
            wp_send_json_error('Faktureringsmail är obligatoriskt');
            return;
        }
        
        if (isset($_POST['billing_reference']) && !empty($_POST['billing_reference'])) {
            update_option('bokforehandel_billing_reference', sanitize_text_field($_POST['billing_reference']));
        } else {
            wp_send_json_error('Fakturareferens är obligatoriskt');
            return;
        }
        
        // Hantera faktureringsadressen som textfält
        if (isset($_POST['billing_address']) && !empty($_POST['billing_address'])) {
            update_option('bokforehandel_billing_address', sanitize_textarea_field($_POST['billing_address']));
        } else {
            wp_send_json_error('Faktureringsadress är obligatoriskt');
            return;
        }
        
        // Pris och faktureringsvillkor
        if (isset($_POST['billing_price_option'])) {
            $price_option = sanitize_text_field($_POST['billing_price_option']);
            if (!in_array($price_option, ['monthly', 'yearly'])) {
                wp_send_json_error('Välj ett giltigt prisalternativ');
                return;
            }
            update_option('bokforehandel_billing_price_option', $price_option);
        }
        
        // Bekräftelse och ansvarsfriskrivning
        if (isset($_POST['billing_agree'])) {
            update_option('bokforehandel_billing_agree', '1');
        } else {
            wp_send_json_error('Du måste godkänna fakturavillkoren');
            return;
        }
        
        if (isset($_POST['disclaimer_agreed'])) {
            update_option('bokforehandel_disclaimer_agreed', '1');
        } else {
            wp_send_json_error('Du måste godkänna användarvillkoren');
            return;
        }
        
        // API-Url
        $api_url = isset($_POST['api_url']) ? sanitize_text_field($_POST['api_url']) : '';
        if (empty($api_url)) {
            $api_url = $this->get_api_url();
        } else {
            // Trimma bort avslutande slash
            $api_url = rtrim($api_url, '/');
        }
        update_option('bokforehandel_api_url', $api_url);
        
        // Sanera och spara API-nycklar
        $bl_user_key = '';
        if (isset($_POST['bl_user_key'])) {
            $bl_user_key = sanitize_text_field($_POST['bl_user_key']);
            update_option('bokforehandel_bl_user_key', $bl_user_key);
        }
        
        // Uppdatera användarinformation i API:et
        $user_id = get_option('bokforehandel_user_id', '');
        
        if (!empty($user_id)) {
            // Om vi har ett user_id, använd det för att uppdatera användaren
            $result = $this->update_user_with_api($user_id);
            
            // Skicka en notifiering med fakturauppgifter till API:et oavsett resultat
            $this->send_billing_notification();
            
            if ($result['status'] === 'success') {
                wp_send_json_success('Företagsinformationen har sparats och synkroniserats med BL');
            } else {
                wp_send_json_error('Företagsinformationen sparades lokalt men kunde inte synkroniseras med BL. Kontrollera att du aktiverat integrationen hos Bjorn Lunden!: ' . $result['message']);
            }
        } else {
            // Om vi inte har ett user_id, registrera användaren
            $result = $this->register_user_with_api();
            
            // Skicka en notifiering med fakturauppgifter till API:et oavsett resultat
            $this->send_billing_notification();
            
            if ($result['status'] === 'success') {
                // Hantera multibutiksläge i meddelandet
                $success_message = 'Företagsinformationen har sparats och registrerats med BL';
                if (isset($result['multibutik_mode']) && $result['multibutik_mode'] === true) {
                    $success_message = 'Företagsinformationen har sparats lokalt. ' . $result['message'];
                }
                wp_send_json_success($success_message);
            } else {
                wp_send_json_error('Företagsinformationen sparades lokalt men kunde inte registreras med BL: ' . $result['message']);
            }
        }
    }
    
    /**
     * AJAX: Återställ alla inställningar
     */
    public function ajax_reset_settings() {
        // Kontrollera AJAX-nonce
        if (!$this->verify_ajax_security('bokforehandel_nonce')) {
            return;
        }
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Du har inte behörighet att utföra denna åtgärd'));
            return;
        }
        
        // Spara användarregistreringsinformation
        $bl_user_key = get_option('bokforehandel_bl_user_key', '');
        $bl_client_secret = get_option('bokforehandel_bl_client_secret', '');
        $organization_number = get_option('bokforehandel_organization_number', '');
        $company_name = get_option('bokforehandel_company_name', '');
        $store_name = get_option('bokforehandel_store_name', '');
        $integration_status = get_option('bokforehandel_integration_status', 'off');
        
        // Lista över alla inställningar att återställa (exklusive användaridentifikation)
        $options_to_reset = array(
            'bokforehandel_wc_api_url',
            'bokforehandel_wc_consumer_key',
            'bokforehandel_wc_consumer_secret',
            'bokforehandel_sales_account',
            'bokforehandel_vat_account',
            'bokforehandel_bank_account',
            'bokforehandel_receivables_account',
            'bokforehandel_fee_account',
            'bokforehandel_input_vat_account',
            'bokforehandel_payment_accounts',
            'bokforehandel_payment_settlements',
            'bokforehandel_journal_id',
            'bokforehandel_klarna_journal_id',
            'bokforehandel_stripe_journal_id',
            'bokforehandel_klarna_enabled',
            'bokforehandel_klarna_api_username',
            'bokforehandel_klarna_api_password',
            'bokforehandel_stripe_enabled',
            'bokforehandel_stripe_api_key',
            'bokforehandel_sync_enabled',
            'bokforehandel_sync_frequency',
            'bokforehandel_sync_period',
            'bokforehandel_upload_receipts',
            'bokforehandel_upload_settlements',
            'bokforehandel_billing_email',
            'bokforehandel_billing_reference',
            'bokforehandel_billing_address',
            'bokforehandel_billing_price_option',
            'bokforehandel_billing_agree',
            'bokforehandel_disclaimer_agreed',
            'bokforehandel_last_run_id',
            'bokforehandel_last_run_time'
        );
        
        // Ta bort alla options
        foreach ($options_to_reset as $option) {
            delete_option($option);
        }
        
        // Återställ användaridentifikation
        update_option('bokforehandel_bl_user_key', $bl_user_key);
        update_option('bokforehandel_bl_client_secret', $bl_client_secret);
        update_option('bokforehandel_organization_number', $organization_number);
        update_option('bokforehandel_company_name', $company_name);
        update_option('bokforehandel_store_name', $store_name);
        update_option('bokforehandel_integration_status', $integration_status);
        
        // Återställ standardvärden för konton
        update_option('bokforehandel_sales_account', '3001');
        update_option('bokforehandel_vat_account', '2610');
        update_option('bokforehandel_bank_account', '1930');
        update_option('bokforehandel_receivables_account', '1510');
        update_option('bokforehandel_fee_account', '6570');
        update_option('bokforehandel_input_vat_account', '2640');
        update_option('bokforehandel_journal_id', 'W');
        
        wp_send_json_success(array('message' => 'Inställningar har återställts'));
    }

    /**
     * AJAX: Spara systeminställningar
     */
    public function ajax_save_system_settings() {
        // Kontrollera AJAX-nonce
        if (!$this->verify_ajax_security('bokforehandel_nonce')) {
            return;
        }
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Du har inte behörighet att utföra denna åtgärd'));
            return;
        }
        
        // Hämta och spara inställning för att behålla data vid avinstallation
        $keep_data = isset($_POST['keep_data_on_uninstall']) && $_POST['keep_data_on_uninstall'];
        update_option('bokforehandel_keep_data_on_uninstall', $keep_data);
        
        error_log('Bokför Ehandel: Systeminställningar sparade - Behåll data: ' . ($keep_data ? 'Ja' : 'Nej'));
        
        wp_send_json_success(array('message' => 'Systeminställningar har sparats'));
    }

    /**
     * Skickar en notifiering med fakturauppgifter till API:et
     * 
     * @return array Resultat från API-anropet
     */
    private function send_billing_notification() {
        // Hämta nödvändiga data
        $company_name = get_option('bokforehandel_company_name', '');
        $organization_number = get_option('bokforehandel_organization_number', '');
        $integration_status = get_option('bokforehandel_integration_status', 'off');
        
        // Faktureringsinformation
        $billing_email = get_option('bokforehandel_billing_email', '');
        $billing_reference = get_option('bokforehandel_billing_reference', '');
        $billing_address = get_option('bokforehandel_billing_address', '');
        $billing_plan = get_option('bokforehandel_billing_price_option', 'monthly');
        
        // Kontrollera om vi har de nödvändiga uppgifterna
        if (empty($organization_number) || empty($company_name) || empty($billing_email) || empty($billing_reference)) {
            error_log('Bokfor Ehandel: Kan inte skicka faktureringsuppgifter - saknar obligatoriska fält');
            return array(
                'status' => 'error',
                'message' => 'Saknar nödvändiga fakturauppgifter'
            );
        }
        
        // Förbered data för API-anrop
        $api_data = array(
            'organization_number' => $organization_number,
            'company_name' => $company_name,
            'store_name' => get_option('bokforehandel_store_name', ''),
            'billing_email' => $billing_email,
            'billing_reference' => $billing_reference,
            'billing_address' => $billing_address,
            'billing_plan' => $billing_plan,
            'integration_status' => $integration_status
        );
        
        // Anropa API:et
        $api_url = BOKFOREHANDEL_API_URL . '/api/billing_notification';
        
        $response = wp_remote_post($api_url, array(
            'method' => 'POST',
            'timeout' => 10,
            'redirection' => 5,
            'httpversion' => '1.0',
            'blocking' => true,
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-API-Key' => get_option('bokforehandel_bl_user_key', '')
            ),
            'body' => json_encode($api_data),
            'cookies' => array()
        ));
        
        // Kontrollera svaret
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            error_log('Bokfor Ehandel: API-fel vid faktureringsnotifiering: ' . $error_message);
            
            return array(
                'status' => 'error',
                'message' => 'Kunde inte skicka faktureringsnotifiering: ' . $error_message
            );
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if ($status_code === 200 && isset($body['status']) && $body['status'] === 'success') {
            return array(
                'status' => 'success',
                'message' => 'Faktureringsnotifiering skickad'
            );
        } else {
            $error_msg = isset($body['message']) ? $body['message'] : 'Okänt fel';
            error_log('Bokfor Ehandel: API-fel vid faktureringsnotifiering: ' . $error_msg);
            
            return array(
                'status' => 'error',
                'message' => 'Kunde inte skicka faktureringsnotifiering: ' . $error_msg
            );
        }
    }

    /**
     * AJAX: Spara bokföringsinställningar
     */
    public function ajax_save_accounting() {
        try {
            // Kontrollera AJAX-nonce
            if (!$this->verify_ajax_security('bokforehandel_nonce')) {
                return;
            }

            if (!current_user_can('manage_options')) {
                wp_send_json_error('Du har inte behörighet att utföra denna åtgärd');
                return;
            }
        
        // Spara standard bokföringskonton och verifiera att de sparades korrekt
        $save_errors = array();
        $saved_accounts = array();
        
        if (isset($_POST['sales_account'])) {
            $value = sanitize_text_field($_POST['sales_account']);
            update_option('bokforehandel_sales_account', $value);
            $saved = get_option('bokforehandel_sales_account', '');
            if ($saved !== $value) {
                $save_errors[] = 'Försäljningskonto (sales_account)';
            } else {
                $saved_accounts['sales_account'] = $value;
            }
        }
        
        if (isset($_POST['vat_account'])) {
            $value = sanitize_text_field($_POST['vat_account']);
            update_option('bokforehandel_vat_account', $value);
            $saved = get_option('bokforehandel_vat_account', '');
            if ($saved !== $value) {
                $save_errors[] = 'Momskonto (vat_account)';
            } else {
                $saved_accounts['vat_account'] = $value;
            }
        }
        
        if (isset($_POST['bank_account'])) {
            $value = sanitize_text_field($_POST['bank_account']);
            update_option('bokforehandel_bank_account', $value);
            $saved = get_option('bokforehandel_bank_account', '');
            if ($saved !== $value) {
                $save_errors[] = 'Bankkonto (bank_account)';
            } else {
                $saved_accounts['bank_account'] = $value;
            }
        }
        
        if (isset($_POST['receivables_account'])) {
            $value = sanitize_text_field($_POST['receivables_account']);
            update_option('bokforehandel_receivables_account', $value);
            $saved = get_option('bokforehandel_receivables_account', '');
            if ($saved !== $value) {
                $save_errors[] = 'Kundfordringar (receivables_account)';
            } else {
                $saved_accounts['receivables_account'] = $value;
            }
        }
        
        if (isset($_POST['fee_account'])) {
            $value = sanitize_text_field($_POST['fee_account']);
            update_option('bokforehandel_fee_account', $value);
            $saved = get_option('bokforehandel_fee_account', '');
            if ($saved !== $value) {
                $save_errors[] = 'Avgiftskonto (fee_account)';
            } else {
                $saved_accounts['fee_account'] = $value;
            }
        }
        
        if (isset($_POST['input_vat_account'])) {
            $value = sanitize_text_field($_POST['input_vat_account']);
            update_option('bokforehandel_input_vat_account', $value);
            $saved = get_option('bokforehandel_input_vat_account', '');
            if ($saved !== $value) {
                $save_errors[] = 'Ingående moms (input_vat_account)';
            } else {
                $saved_accounts['input_vat_account'] = $value;
            }
        }
        
        // Om några konton misslyckades att spara, returnera felmeddelande
        if (!empty($save_errors)) {
            $error_message = 'Följande konton kunde inte sparas korrekt: ' . implode(', ', $save_errors);
            error_log('Bokfor Ehandel: ' . $error_message);
            wp_send_json_error($error_message . '. Kontrollera WordPress-loggarna för mer information.');
            return;
        }
        
        // Logga att konton sparades korrekt
        error_log('Bokfor Ehandel: Följande konton sparades korrekt: ' . implode(', ', array_keys($saved_accounts)));
        
        // Spara betalningsmetodskonton
        if (isset($_POST['payment_accounts']) && is_array($_POST['payment_accounts'])) {
            $payment_accounts = array();
            foreach ($_POST['payment_accounts'] as $method => $account) {
                $payment_accounts[sanitize_text_field($method)] = sanitize_text_field($account);
            }
            update_option('bokforehandel_payment_accounts', $payment_accounts);
        }
        
        // Spara avstämningskonton
        if (isset($_POST['payment_settlements']) && is_array($_POST['payment_settlements'])) {
            $payment_settlements = array();
            foreach ($_POST['payment_settlements'] as $method => $use_settlement) {
                $payment_settlements[sanitize_text_field($method)] = !empty($use_settlement) ? '1' : '0';
            }
            update_option('bokforehandel_payment_settlements', $payment_settlements);
        }

        // Spara journal IDs (bokföringsserier)
        if (isset($_POST['accounts']) && is_array($_POST['accounts'])) {
            $accounts = array();
            foreach ($_POST['accounts'] as $key => $value) {
                $accounts[sanitize_text_field($key)] = sanitize_text_field($value);
            }
            update_option('bokforehandel_accounts', $accounts);
        }
        
        // Hantera Klarna-inställningar
        $klarna_enabled = isset($_POST['klarna_enabled']) && $_POST['klarna_enabled'] === 'true';
        update_option('bokforehandel_klarna_enabled', $klarna_enabled);
        
        if ($klarna_enabled) {
            if (isset($_POST['klarna_api_username'])) {
                update_option('bokforehandel_klarna_api_username', sanitize_text_field($_POST['klarna_api_username']));
            }
            
            if (isset($_POST['klarna_api_password'])) {
                update_option('bokforehandel_klarna_api_password', sanitize_text_field($_POST['klarna_api_password']));
            }
        }
        
        // Hantera Stripe-inställningar
        $stripe_enabled = isset($_POST['stripe_enabled']) && $_POST['stripe_enabled'] === 'true';
        update_option('bokforehandel_stripe_enabled', $stripe_enabled);
        
        if ($stripe_enabled) {
            if (isset($_POST['stripe_api_key'])) {
                update_option('bokforehandel_stripe_api_key', sanitize_text_field($_POST['stripe_api_key']));
            }
        }
        
        // Dokumentuppladdningsinställningar
        $upload_receipts = isset($_POST['upload_receipts']) && $_POST['upload_receipts'] === 'true';
        update_option('bokforehandel_upload_receipts', $upload_receipts);
        
        $upload_settlements = isset($_POST['upload_settlements']) && $_POST['upload_settlements'] === 'true';
        update_option('bokforehandel_upload_settlements', $upload_settlements);
        
        // Uppdatera användarinformation i API:et om vi har ett user_id
        $user_id = get_option('bokforehandel_user_id', '');
        if (!empty($user_id)) {
            $result = $this->update_user_with_api($user_id);
            if (isset($result['status']) && $result['status'] === 'success') {
                wp_send_json_success('Bokföringsinställningarna har sparats och synkroniserats med BL');
            } else {
                wp_send_json_error('Bokföringsinställningarna sparades lokalt men kunde inte synkroniseras med BL: ' . (isset($result['message']) ? $result['message'] : 'Okänt fel'));
            }
        } else {
            wp_send_json_success('Bokföringsinställningarna har sparats lokalt. Inga ändringar skickades till BL.');
        }

        } catch (Exception $e) {
            error_log('Bokfor Ehandel: Fel i ajax_save_accounting: ' . $e->getMessage());
            error_log('Bokfor Ehandel: Stack trace: ' . $e->getTraceAsString());
            wp_send_json_error('Ett oväntat fel uppstod vid sparande av bokföringsinställningar. Fel har loggats.');
        }
    }

    /**
     * AJAX: Spara synkroniseringsinställningar
     */
    public function ajax_save_sync() {
        // Kontrollera AJAX-nonce
        if (!$this->verify_ajax_security('bokforehandel_nonce')) {
            return;
        }
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Du har inte behörighet att utföra denna åtgärd');
            return;
        }
        
        // Spara synkroniseringsinställningar
        $sync_enabled = isset($_POST['sync_enabled']) && $_POST['sync_enabled'] == 'true' ? 'yes' : 'no';
        update_option('bokforehandel_sync_enabled', $sync_enabled);
        
        if (isset($_POST['sync_frequency'])) {
            $sync_frequency = sanitize_text_field($_POST['sync_frequency']);
            if (!in_array($sync_frequency, ['daily', 'weekly', 'monthly'])) {
                wp_send_json_error('Välj en giltig synkroniseringsfrekvens');
                return;
            }
            update_option('bokforehandel_sync_frequency', $sync_frequency);
        }
        
        if (isset($_POST['sync_period'])) {
            $sync_period = sanitize_text_field($_POST['sync_period']);
            if (!in_array($sync_period, ['lastday', 'lastweek', 'lastmonth', 'custom'])) {
                wp_send_json_error('Välj en giltig synkroniseringsperiod');
                return;
            }
            update_option('bokforehandel_sync_period', $sync_period);
        }
        
        // Om vi har en custom period, spara startdatum och slutdatum
        if (isset($_POST['sync_period']) && $_POST['sync_period'] == 'custom') {
            if (isset($_POST['sync_start_date'])) {
                update_option('bokforehandel_sync_start_date', sanitize_text_field($_POST['sync_start_date']));
            }
            
            if (isset($_POST['sync_end_date'])) {
                update_option('bokforehandel_sync_end_date', sanitize_text_field($_POST['sync_end_date']));
            }
        }

        // Hantera integration_status (på/av för automatisk bokföring)
        if (isset($_POST['integration_status'])) {
            $new_integration_status = sanitize_text_field($_POST['integration_status']);
            if (!in_array($new_integration_status, ['on', 'off'])) {
                wp_send_json_error('Välj en giltig integrationsstatus');
                return;
            }

            $old_integration_status = get_option('bokforehandel_integration_status', 'off');

            // Spara inställningen lokalt
            update_option('bokforehandel_integration_status', $new_integration_status);

            // Om statusen har ändrats, skicka webhook till API
            if ($old_integration_status !== $new_integration_status) {
                $this->send_integration_webhook($new_integration_status);
                error_log("Bokfor Ehandel: Integration status ändrad från '$old_integration_status' till '$new_integration_status' och webhook skickad");
            }
        }

        // Uppdatera användarinformation i API:et om vi har ett user_id
        $user_id = get_option('bokforehandel_user_id', '');
        if (!empty($user_id)) {
            $result = $this->update_user_with_api($user_id);
            if (isset($result['status']) && $result['status'] === 'success') {
                wp_send_json_success('Synkroniseringsinställningarna har sparats och synkroniserats med BL');
            } else {
                wp_send_json_error('Synkroniseringsinställningarna sparades lokalt men kunde inte synkroniseras med BL: ' . (isset($result['message']) ? $result['message'] : 'Okänt fel'));
            }
        } else {
            wp_send_json_success('Synkroniseringsinställningarna har sparats lokalt. Inga ändringar skickades till BL.');
        }
    }

    /**
     * Verifierar att AJAX-anrop är giltiga med nonce
     *
     * @param string $nonce_action Nonce-åtgärd att verifiera mot
     * @param bool $die Om anropet ska avbrytas vid misslyckad verifiering
     * @return bool Om verifieringen lyckades eller inte
     */
    private function verify_ajax_security($nonce_action, $die = true) {
        // Kontrollera AJAX-nonce från POST
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], $nonce_action)) {
            if ($die) {
                wp_send_json_error('Säkerhetsverifiering misslyckades');
                wp_die();
            }
            return false;
        }
        
        // Kontrollera behörighet
        if (!current_user_can('manage_options')) {
            if ($die) {
                wp_send_json_error('Du har inte behörighet att utföra denna åtgärd');
                wp_die();
            }
            return false;
        }
        
        return true;
    }

    /**
     * Kontrollerar om det finns uppdateringar tillgängliga för pluginet
     */
    public function check_for_updates($transient) {
        if (empty($transient->checked)) {
            return $transient;
        }

        $plugin_slug = plugin_basename(BOKFOREHANDEL_PLUGIN_FILE);
        $current_version = BOKFOREHANDEL_VERSION;

        // Add cache-busting parameter to force fresh check
        $remote_url = 'https://bokforehandel.se/wordpress/version.json?t=' . time();
        $remote_response = wp_remote_get($remote_url, array(
            'timeout' => 10,
            'sslverify' => true,
            'user-agent' => 'WordPress/' . get_bloginfo('version') . '; ' . home_url()
        ));

        // Log the update check for debugging
        error_log("Bokfor Ehandel: Checking for updates from {$remote_url}");
        error_log("Bokfor Ehandel: Current version: {$current_version}");
        
        if (!is_wp_error($remote_response) && 200 === wp_remote_retrieve_response_code($remote_response)) {
            $remote_body = wp_remote_retrieve_body($remote_response);
            $remote_data = json_decode($remote_body, true);

            if ($remote_data && isset($remote_data['version'])) {
                error_log("Bokfor Ehandel: Remote version: {$remote_data['version']}");

                if (version_compare($current_version, $remote_data['version'], '<')) {
                    error_log("Bokfor Ehandel: Update available! {$current_version} -> {$remote_data['version']}");

                    $res = new stdClass();
                    $res->slug = 'bokfor-ehandel';
                    $res->plugin = $plugin_slug;
                    $res->new_version = $remote_data['version'];
                    $res->tested = isset($remote_data['tested']) ? $remote_data['tested'] : '6.8';
                    $res->requires = isset($remote_data['requires']) ? $remote_data['requires'] : '6.3';
                    $res->package = 'https://bokforehandel.se/wordpress/bokfor-ehandel.zip';
                    $res->url = isset($remote_data['homepage']) ? $remote_data['homepage'] : 'https://www.bokforehandel.se';
                    $res->download_link = $res->package;

                    $transient->response[$plugin_slug] = $res;
                    error_log("Bokfor Ehandel: Update response added to transient");
                } else {
                    error_log("Bokfor Ehandel: No update available (current version is latest)");
                }
            } else {
                error_log("Bokfor Ehandel: Invalid response data: " . $remote_body);
            }
        } else {
            $error_code = is_wp_error($remote_response) ? $remote_response->get_error_code() : wp_remote_retrieve_response_code($remote_response);
            $error_message = is_wp_error($remote_response) ? $remote_response->get_error_message() : 'HTTP Error';
            error_log("Bokfor Ehandel: Update check failed: {$error_code} - {$error_message}");
        }
        
        return $transient;
    }

    /**
     * Force clear WordPress plugin update cache
     * Call this function to force immediate update check
     */
    public function force_update_check() {
        // Clear WordPress transient cache
        delete_site_transient('update_plugins');
        delete_site_transient('update_themes');
        delete_site_transient('update_core');

        // Force WordPress to check for updates immediately
        wp_version_check();
        wp_update_plugins();

        error_log("Bokfor Ehandel: Update cache cleared and forced check initiated");
    }

    /**
     * AJAX: Force update check (for admin use)
     */
    public function ajax_force_update_check() {
        try {
            // Verify nonce and capabilities
            if (!$this->verify_ajax_security('bokforehandel_nonce')) {
                return;
            }

            if (!current_user_can('manage_options')) {
                wp_send_json_error('Du har inte behörighet att utföra denna åtgärd');
                return;
            }

            // Force the update check
            $this->force_update_check();

            // Get current version and check for updates
            $current_version = BOKFOREHANDEL_VERSION;
            $remote_url = 'https://bokforehandel.se/wordpress/version.json?t=' . time();
            $remote_response = wp_remote_get($remote_url, array(
                'timeout' => 10,
                'sslverify' => true
            ));

            if (!is_wp_error($remote_response) && 200 === wp_remote_retrieve_response_code($remote_response)) {
                $remote_data = json_decode(wp_remote_retrieve_body($remote_response), true);
                $latest_version = isset($remote_data['version']) ? $remote_data['version'] : 'Unknown';

                $update_available = version_compare($current_version, $latest_version, '<');

                wp_send_json_success(array(
                    'current_version' => $current_version,
                    'latest_version' => $latest_version,
                    'update_available' => $update_available,
                    'message' => $update_available
                        ? "Uppdatering tillgänglig: {$current_version} → {$latest_version}"
                        : "Du har senaste versionen: {$current_version}"
                ));
            } else {
                wp_send_json_error('Kunde inte kontrollera uppdateringar');
            }

        } catch (Exception $e) {
            error_log('Bokfor Ehandel: Force update check error: ' . $e->getMessage());
            wp_send_json_error('Ett fel uppstod vid uppdateringskontroll');
        }
    }

    /**
     * AJAX: Health check endpoint
     */
    public function ajax_health_check() {
        try {
            // Kontrollera åtkomst
            if (!current_user_can('manage_options')) {
                wp_send_json_error('Otillåten åtkomst');
                exit;
            }

            // Kontrollera nonce
            if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'bokforehandel_nonce')) {
                wp_send_json_error('Säkerhetsverifiering misslyckades');
                exit;
            }

            $user_id = get_option('bokforehandel_user_id', '');
            $health_status = array(
                'timestamp' => current_time('mysql'),
                'plugin_version' => BOKFOREHANDEL_VERSION,
                'wordpress_version' => get_bloginfo('version'),
                'php_version' => PHP_VERSION,
                'database_connection' => $this->test_database_connection(),
                'api_connection' => $user_id ? $this->test_api_connection($user_id) : 'not_configured',
                'last_sync' => get_option('bokforehandel_last_run_time', 'never'),
                'cache_status' => $this->get_cache_status(),
                'memory_usage' => memory_get_usage(true),
                'peak_memory' => memory_get_peak_usage(true)
            );

            wp_send_json_success($health_status);

        } catch (Exception $e) {
            error_log('Bokfor Ehandel: Health check error: ' . $e->getMessage());
            wp_send_json_error('Health check failed');
        }
    }

    private function test_database_connection() {
        global $wpdb;
        try {
            $wpdb->get_var('SELECT 1');
            return 'ok';
        } catch (Exception $e) {
            return 'error: ' . $e->getMessage();
        }
    }

    private function test_api_connection($user_id) {
        try {
            $status = $this->check_bl_integration_status($user_id);
            return $status === 'on' ? 'ok' : 'inactive';
        } catch (Exception $e) {
            return 'error: ' . $e->getMessage();
        }
    }

    private function get_cache_status() {
        // Check if transients are working
        $test_key = 'bokfore_test_cache_' . time();
        set_transient($test_key, 'test', 10);
        $result = get_transient($test_key) === 'test' ? 'ok' : 'error';
        delete_transient($test_key);
        return $result;
    }

    /**
     * Tillhandahåller information om pluginet för uppdateringsdialogen
     */
    public function plugin_info($res, $action, $args) {
        if ('plugin_information' !== $action || 'bokfor-ehandel' !== $args->slug) {
            return $res;
        }
        
        $remote_url = 'https://bokforehandel.se/wordpress/version.json?t=' . time();
        $remote_response = wp_remote_get($remote_url, array(
            'timeout' => 10,
            'sslverify' => true,
            'user-agent' => 'WordPress/' . get_bloginfo('version') . '; ' . home_url()
        ));
        
        if (!is_wp_error($remote_response) && 200 === wp_remote_retrieve_response_code($remote_response)) {
            $remote_body = wp_remote_retrieve_body($remote_response);
            $remote_data = json_decode($remote_body, true);
            
            if ($remote_data) {
                $res = new stdClass();
                $res->name = 'Bokför E-handel';
                $res->slug = 'bokfor-ehandel';
                $res->version = isset($remote_data['version']) ? $remote_data['version'] : BOKFOREHANDEL_VERSION;
                $res->tested = isset($remote_data['tested']) ? $remote_data['tested'] : '6.4';
                $res->requires = isset($remote_data['requires']) ? $remote_data['requires'] : '5.0';
                $res->author = 'Ideäng Productions';
                $res->author_profile = 'https://www.bokforehandel.se';
                $res->download_link = 'https://bokforehandel.se/wordpress/bokfor-ehandel.zip';
                $res->trunk = 'https://bokforehandel.se/wordpress/bokfor-ehandel.zip';
                $res->last_updated = isset($remote_data['last_updated']) ? $remote_data['last_updated'] : '2025-04-01';
                $res->sections = array(
                    'description' => isset($remote_data['sections']['description']) ? $remote_data['sections']['description'] : 'Koppla din webbutik till BL Bokföring',
                    'changelog' => isset($remote_data['sections']['changelog']) ? $remote_data['sections']['changelog'] : 'Se readme.txt för ändringslogg',
                );
                
                return $res;
            }
        }
        
        return $res;
    }
}

// Lägg till efter plugin header-kommentaren, innan första if-satsen:
// Lägg till inställningslänk i plugins-listan
add_filter('plugin_action_links_' . plugin_basename(__FILE__), function($links) {
    $settings_link = '<a href="' . admin_url('admin.php?page=bokforehandel-company') . '">Inställningar</a>';
    array_unshift($links, $settings_link);
    return $links;
});

// Modifiera aktiveringsmeddelandet
add_action('activated_plugin', function($plugin) {
    if($plugin == plugin_basename(__FILE__)) {
        $html = '<div class="updated notice is-dismissible" style="padding: 20px; font-size: 14px;">';
        $html .= '<p><strong>Bokför Ehandel har aktiverats!</strong></p>';
        $html .= '<p>För att komma igång, gå till <a href="' . admin_url('admin.php?page=bokforehandel-company') . '">Inställningar</a> och konfigurera din integration.</p>';
        $html .= '</div>';
        set_transient('bokforehandel_activation_notice', $html, 5);
    }
});

// Visa aktiveringsmeddelandet
add_action('admin_notices', function() {
    if ($notice = get_transient('bokforehandel_activation_notice')) {
        echo $notice;
        delete_transient('bokforehandel_activation_notice');
    }
});

// Aktiveringshook för att initiera pluginet och skicka installation ping
register_activation_hook(__FILE__, 'bokforehandel_activate_plugin');

function bokforehandel_activate_plugin() {
    // Säkerställ att WordPress kan hantera inställningarna
    // Genom att tvinga fram en tidig initiering av settings
    add_action('admin_init', function() {
        // Denna funktion kommer att köras när admin är redo
        // vilket säkerställer att settings är korrekt laddade
    }, 5);

    // Sätt en flagga för att indikera att pluginet har aktiverats
    update_option('bokforehandel_activated', true);

    // Anropa klassens activate-metod
    // Skapa en instans om den inte finns ännu (vid aktivering körs init-hooken inte än)
    if (class_exists('Bokfor_Ehandel')) {
        try {
            // Om instansen inte finns, skapa en temporär för aktivering
            if (!isset($GLOBALS['bokforehandel_instance'])) {
                $instance = new Bokfor_Ehandel();
                $instance->activate();
            } else {
                $GLOBALS['bokforehandel_instance']->activate();
            }
        } catch (Exception $e) {
            error_log('Bokför Ehandel - Fel i activate(): ' . $e->getMessage());
        }
    }

    // Försök att skicka installation ping (utan att returnera fel)
    try {
        bokforehandel_send_installation_ping();
    } catch (Exception $e) {
        // Logga felet men hindra inte aktiveringen
        error_log('Bokför Ehandel - Fel vid aktivering: ' . $e->getMessage());
    }
}

function bokforehandel_send_installation_ping() {
    // Hämta systeminformation
    global $wp_version;

    $site_url = get_site_url();
    $site_name = get_bloginfo('name');
    $admin_email = get_option('admin_email');
    $plugin_version = BOKFOREHANDEL_VERSION;
    $wordpress_version = $wp_version;
    $php_version = PHP_VERSION;

    // Hämta WooCommerce version om det är installerat
    $woo_version = 'Ej installerat';
    if (class_exists('WooCommerce')) {
        global $woocommerce;
        $woo_version = $woocommerce->version;
    }

    // ÄNDRAT: /api/installation-ping kräver ingen autentisering (ingen dekorator i app.py)
    // Endpoint endast loggar installationsinformation och exponerar ingen känslig data
    // Ta bort kontrollen för bokforehandel_api_key som inte längre finns
    
    // Förbered data för ping
    $ping_data = array(
        'site_url' => $site_url,
        'site_name' => $site_name,
        'plugin_version' => $plugin_version,
        'wordpress_version' => $wordpress_version,
        'php_version' => $php_version,
        'woo_version' => $woo_version,
        'admin_email' => $admin_email,
        'timestamp' => current_time('mysql')
    );

    // Skicka ping till API-servern
    $api_url = BOKFOREHANDEL_API_URL . '/api/installation-ping';

    // ÄNDRAT: Ta bort X-API-Key header eftersom endpoint inte kräver autentisering
    $response = wp_remote_post($api_url, array(
        'method' => 'POST',
        'timeout' => 15,
        'headers' => array(
            'Content-Type' => 'application/json',
            // Ingen X-API-Key header - endpoint kräver ingen autentisering
        ),
        'body' => json_encode($ping_data),
        'sslverify' => true,
        'user-agent' => 'Bokfor-Ehandel-Plugin/' . $plugin_version
    ));

    // Logga resultatet (endast för felsökning, kan tas bort i produktion)
    if (is_wp_error($response)) {
        error_log('Bokför Ehandel - Kunde inte skicka installation ping: ' . $response->get_error_message());
    } else {
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);

        if ($response_code === 200) {
            error_log('Bokför Ehandel - Installation ping skickad framgångsrikt');
            update_option('bokforehandel_installation_ping_sent', true);
        } else {
            error_log('Bokför Ehandel - Installation ping misslyckades. Status: ' . $response_code . ', Body: ' . $response_body);
        }
    }
}

// Lägg till en cron-jobb för att skicka ping om direktanrop misslyckas
add_action('wp', 'bokforehandel_schedule_installation_ping');

function bokforehandel_schedule_installation_ping() {
    // Kontrollera om vi redan har skickat ping för denna installation
    $ping_sent = get_option('bokforehandel_installation_ping_sent', false);

    if (!$ping_sent) {
        // Schemalägg ping att skickas inom 5 minuter
        if (!wp_next_scheduled('bokforehandel_send_delayed_ping')) {
            wp_schedule_single_event(time() + 300, 'bokforehandel_send_delayed_ping');
        }
    }
}

// Hook för det fördröjda pinget
add_action('bokforehandel_send_delayed_ping', 'bokforehandel_send_installation_ping');

// Felsökningsfunktion för att hjälpa till att identifiera problem
// Lägg till denna som en admin action för enkel felsökning
add_action('admin_post_bokforehandel_debug_info', 'bokforehandel_debug_info');

function bokforehandel_debug_info() {
    if (!current_user_can('manage_options')) {
        wp_die('Du har inte behörighet att visa denna sida.');
    }

    echo '<div class="wrap">';
    echo '<h1>Bokför Ehandel - Felsökningsinformation</h1>';

    echo '<h2>Systeminformation</h2>';
    echo '<table class="widefat striped">';
    echo '<tr><td>WordPress version:</td><td>' . get_bloginfo('version') . '</td></tr>';
    echo '<tr><td>PHP version:</td><td>' . PHP_VERSION . '</td></tr>';
    echo '<tr><td>Plugin version:</td><td>' . BOKFOREHANDEL_VERSION . '</td></tr>';
    echo '<tr><td>WooCommerce status:</td><td>' . (class_exists('WooCommerce') ? 'Installerat (v' . WC()->version . ')' : 'Ej installerat') . '</td></tr>';
    echo '<tr><td>Plugin aktiverad:</td><td>' . (get_option('bokforehandel_activated', false) ? 'Ja' : 'Nej') . '</td></tr>';
    echo '</table>';

    echo '<h2>Inställningar</h2>';
    echo '<table class="widefat striped">';
    $settings = array(
        'bokforehandel_company_name' => 'Företagsnamn',
        'bokforehandel_api_key' => 'API-nyckel',
        'bokforehandel_integration_status' => 'Integrationsstatus'
    );

    foreach ($settings as $key => $label) {
        $value = get_option($key, 'Ej satt');
        if ($key === 'bokforehandel_api_key') {
            $value = !empty($value) ? 'Satt (' . strlen($value) . ' tecken)' : 'Ej satt';
        }
        echo '<tr><td>' . $label . ':</td><td>' . esc_html($value) . '</td></tr>';
    }
    echo '</table>';

    echo '<h2>Senaste fel (från WordPress debug log)</h2>';
    $debug_log = WP_CONTENT_DIR . '/debug.log';
    if (file_exists($debug_log)) {
        $lines = file($debug_log);
        $recent_errors = array_slice($lines, -20); // Senaste 20 raderna
        $bokfor_errors = array_filter($recent_errors, function($line) {
            return strpos($line, 'Bokför Ehandel') !== false;
        });

        if (!empty($bokfor_errors)) {
            echo '<pre style="background: #f1f1f1; padding: 10px; overflow: auto;">';
            foreach ($bokfor_errors as $error) {
                echo esc_html($error);
            }
            echo '</pre>';
        } else {
            echo '<p>Inga Bokför Ehandel-fel hittades i senaste loggarna.</p>';
        }
    } else {
        echo '<p>Debug loggfil hittades inte. Kontrollera att WP_DEBUG_LOG är aktivt i wp-config.php</p>';
    }

    echo '<h2>Aktiveringslogg</h2>';
    $upload_dir = wp_upload_dir();
    $log_dir = $upload_dir['basedir'] . '/bokforehandel-logs';
    $activation_log = $log_dir . '/plugin-activation.log';

    if (file_exists($activation_log)) {
        echo '<pre style="background: #f1f1f1; padding: 10px; overflow: auto;">';
        echo esc_html(file_get_contents($activation_log));
        echo '</pre>';
    } else {
        echo '<p>Ingen aktiveringslogg hittades.</p>';
    }

    echo '<p><a href="' . admin_url('admin.php?page=bokforehandel-company') . '" class="button">Tillbaka till inställningar</a></p>';
    echo '</div>';
}

?>