/**
 * Admin JavaScript för Bokför Ehandel
 */
(function($) {
    'use strict';

    // Globala variabler
    let activeTabIndex = localStorage.getItem('bokforehandel_active_tab') || 0;
    let tabsInitialized = false;
    const cacheExpiry = 5 * 60 * 1000; // 5 minuter i millisekunder
    let apiResponseCache = {};
    
    // Säkerställ att vi har tillgång till antingen bokforehandel eller bokforehandel_params
    // Skapa en enhetlig global variabel för att hantera båda möjligheterna
    const apiParams = {};
    
    // Initiera med standardvärden i fall varken bokforehandel eller bokforehandel_params finns
    if (typeof bokforehandel === 'undefined' && typeof bokforehandel_params === 'undefined') {
        console.error('Varken bokforehandel eller bokforehandel_params är definierade. Detta kan bero på att WordPress admin ajax inte fungerar korrekt.');
    } else {
        // Använd bokforehandel om den finns, annars använd bokforehandel_params
        const source = typeof bokforehandel !== 'undefined' ? bokforehandel : bokforehandel_params;
        
        // Kopiera värdena till vår lokala variabel
        apiParams.ajax_url = source.ajax_url || ajaxurl || '';
        apiParams.nonce = source.nonce || '';
        apiParams.user_id = source.user_id || '';
        
        console.log('API-parametrar initierade:', { 
            har_ajax_url: !!apiParams.ajax_url,
            har_nonce: !!apiParams.nonce,
            har_user_id: !!apiParams.user_id
        });
        
        // Skapa fallback-referenser för bakåtkompatibilitet
        if (typeof bokforehandel === 'undefined') {
            window.bokforehandel = apiParams;
        }
        if (typeof bokforehandel_params === 'undefined') {
            window.bokforehandel_params = apiParams;
        }
    }

    // När dokumentet är klart
    $(document).ready(function() {
        initTabs();
        initFormEvents();
        initRunsPage();
        initRetroSync();
        initManualSync();
        
        // Lägg till lyssnare för testanslutning-knappen
        $("#test-connection").on('click', function() {
            var $button = $(this);
            $button.text('Testar...').prop('disabled', true);
            $('#bl-integration-status').text('WordPress OK. Testar API...');
            
            $.ajax({
                url: apiParams.ajax_url,
                type: 'POST',
                data: {
                    action: 'bokforehandel_test_connection',
                    nonce: apiParams.nonce
                },
                success: function(response) {
                    if (response.success) {
                        $('#bl-integration-status').html('<span class="status-on">Anslutning OK</span>');
                    } else {
                        $('#bl-integration-status').html('<span class="status-error">' + response.data + '</span>');
                    }
                    $button.text('Testa anslutning').prop('disabled', false);
                },
                error: function(xhr, status, error) {
                    $('#bl-integration-status').html('<span class="status-error">AJAX-fel: ' + error + '</span>');
                    $button.text('Testa anslutning').prop('disabled', false);
                }
            });
        });
        
        // Kör endast integrationsstatus om elementet finns på sidan
        if ($('#bl-integration-status').length) {
            checkIntegrationStatus();
            
            // Kontrollera status var 5:e minut istället för var 60:e sekund
            setInterval(checkIntegrationStatus, 5 * 60 * 1000);
        }
    });

    /**
     * Initierar flikarna på ett säkert sätt utan dubbelinitiering
     */
    function initTabs() {
        // Kontrollera om vi är på inställningssidan
        if (window.location.href.indexOf('page=bokforehandel-settings') === -1) {
            return; // Inte på inställningssidan, avsluta funktionen
        }
        
        const $tabs = $('#bokforehandel-tabs');
        
        // Kontrollera om flikarna redan har initierats eller om de inte finns på sidan
        if ($tabs.hasClass('ui-tabs') || !$tabs.length || tabsInitialized) {
            return;
        }
        
        // Förhindra att sidan laddas om i onödan vid klick på adminmenyer
        $('.bokforehandel-tabs-nav a').click(function(e) {
            e.preventDefault();
            activeTabIndex = $(this).parent().index();
            localStorage.setItem('bokforehandel_active_tab', activeTabIndex);
        });

        // Kontrollera om användaren kommer direkt till inställningssidan utan att ha klickat på en flik
        // Detta händer när användaren klickar på "Bokför Ehandel > Inställningar" i vänstermenyn
        if (window.location.href.indexOf('page=bokforehandel-settings') !== -1 && 
            window.location.href.indexOf('tab=') === -1) {
            // Om användaren kommer direkt till inställningssidan, börja alltid med första fliken
            activeTabIndex = 0;
            localStorage.setItem('bokforehandel_active_tab', activeTabIndex);
        }

        // Initiera flikar med sparad aktiv flik
        $tabs.tabs({
            active: parseInt(activeTabIndex),
            activate: function(event, ui) {
                activeTabIndex = ui.newTab.index();
                localStorage.setItem('bokforehandel_active_tab', activeTabIndex);
            }
        });
        
        tabsInitialized = true;
        console.log('Flikar har initierats');
    }

    /**
     * Initierar formulärhändelser och kontroller
     */
    function initFormEvents() {
        // Globala variabler - hämta från WP om de finns
        var apiUrl = (typeof apiParams !== 'undefined' && apiParams.ajax_url) ? apiParams.ajax_url : ajaxurl;
        var nonce = (typeof apiParams !== 'undefined' && apiParams.nonce) ? apiParams.nonce : '';

        // Toggle-knappar för flikar
        $('.bokforehandel-tabs-nav li a').on('click', function(e) {
            e.preventDefault();
            
            var tabId = $(this).attr('href');
            
            // Uppdatera aktiv flik
            $('.bokforehandel-tabs-nav li a').removeClass('ui-state-active');
            $(this).addClass('ui-state-active');
            
            // Visa vald flik
            $('.bokforehandel-tab').hide();
            $(tabId).show();
            
            // Lagra aktiv flik i localStorage
            localStorage.setItem('bokforehandel_active_tab', tabId);
        });
        
        // Spara inställningar för automatisk synkronisering
        // Synkronisera de båda auto-sync kontrollerna
        $('#auto-sync-top, #auto-sync').on('change', function() {
            var isChecked = $(this).is(':checked');
            $('#auto-sync-top, #auto-sync').prop('checked', isChecked);
        });
        
        // Hantera "Kör integration nu"-knappen på inställningssidan
        $('.run-integration-btn').not('#run-retro-sync').click(function(e) {
            e.preventDefault();
            
            // Kontrollera integrationsstatus först
            var statusEl = $('#bl-integration-status');
            if (statusEl.text().indexOf('Kontrollerar') !== -1 || statusEl.find('.status-off').length) {
                showNotice('Integrationsstatus måste vara aktiv för att köra integration. Kontrollera dina inställningar.', 'error');
                return;
            }
            
            // Visa laddningsindikator
            var $button = $(this);
            $button.text('Kör...').prop('disabled', true);
            
            // Skicka AJAX-begäran
            ajaxRequest({
                type: 'POST',
                url: apiParams.ajax_url,
                data: {
                    action: 'bokforehandel_run_integration',
                    nonce: apiParams.nonce,
                    operation: 'all',
                    date_range: 'lastday'
                },
                headers: {
                    'X-User-ID': apiParams.user_id
                },
                success: function(response) {
                    if (response.success) {
                        showNotice('Integration har startats. Du kan följa förloppet på "Körningar"-sidan.', 'success');
                        
                        // Fördröj omdirigeringen lite så användaren kan se meddelandet
                        setTimeout(function() {
                            window.location.href = 'admin.php?page=bokforehandel-runs';
                        }, 1500);
                    } else {
                        showNotice('Fel: ' + response.data, 'error');
                        $button.text('Kör integration nu').prop('disabled', false);
                    }
                },
                error: function(xhr, status, error) {
                    showNotice('Serverfel vid start av integration: ' + error, 'error');
                    $button.text('Kör integration nu').prop('disabled', false);
                }
            });
        });
        
        // Hantera Klarna-inställningar
        $('#klarna-enabled').change(function() {
            if ($(this).is(':checked')) {
                $('.klarna-settings').slideDown(300);
            } else {
                $('.klarna-settings').slideUp(300);
            }
        }).change(); // Trigga direkt
        
        // Hantera Stripe-inställningar
        $('#stripe-enabled').change(function() {
            if ($(this).is(':checked')) {
                $('.stripe-settings').slideDown(300);
            } else {
                $('.stripe-settings').slideUp(300);
            }
        }).change(); // Trigga direkt
        
        // Auto-sync toggle
        $('#auto-sync').change(function() {
            var status = $(this).is(':checked') ? 'Aktiverad' : 'Inaktiverad';
            $(this).closest('td').find('.status-text').text(status);
        });
        
        // Handle payment method account selects
        $('.payment-method-account').on('change', function() {
            var gateway = $(this).data('gateway');
            var val = $(this).val();
            
            // Show/hide custom account input field based on selection
            if (val === 'custom') {
                $('.payment-custom-account[data-gateway="' + gateway + '"]').show();
            } else {
                $('.payment-custom-account[data-gateway="' + gateway + '"]').hide();
            }
        });
        
        // Trigger change on load to show/hide appropriate fields
        $('.payment-method-account').each(function() {
            $(this).trigger('change');
        });
        
        // Validera obligatoriska fält
        function validateRequired(form) {
            let valid = true;
            form.find('input[required]').each(function() {
                if ($(this).val() === '') {
                    $(this).addClass('bokforehandel-required');
                    valid = false;
                } else {
                    $(this).removeClass('bokforehandel-required');
                }
            });
            return valid;
        }
        
        // Ta bort markering när användaren börjar skriva
        $(document).on('input', 'input.bokforehandel-required', function() {
            if ($(this).val() !== '') {
                $(this).removeClass('bokforehandel-required');
            }
        });
        
        // Hantera anpassat datumintervall
        $('#date-range').change(function() {
            if ($(this).val() === 'custom') {
                $('#custom-date-range').slideDown(300);
            } else {
                $('#custom-date-range').slideUp(300);
            }
        }).change(); // Trigga direkt
        
        // Spara inställningar per flik
        $('.save-tab-settings').on('click', function() {
            var $button = $(this);
            $button.text('Sparar...').prop('disabled', true);

            var formData = new FormData($('#bokforehandel-settings-form')[0]);
            formData.append('action', 'bokforehandel_save_settings');
            formData.append('nonce', apiParams.nonce); // Se till att nonce är definierat i din PHP-kod

            $.ajax({
                url: apiParams.ajax_url, // Se till att ajax_url är definierat i PHP
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                success: function(response) {
                    $button.text('Spara inställningar').prop('disabled', false);
                    if (response.success) {
                        alert('Inställningarna har sparats!');
                        console.log('Sparade data:', response);
                    } else {
                        alert('Fel vid sparande: ' + (response.data || 'Okänt fel'));
                        console.error('Fel:', response);
                    }
                },
                error: function(xhr, status, error) {
                    $button.text('Spara inställningar').prop('disabled', false);
                    alert('AJAX-fel: ' + error);
                    console.error('AJAX-fel:', xhr.responseText);
                }
            });
        });

    }
    
    /**
     * Initiera körningssidan
     */
    function initRunsPage() {
        // Kontrollera om vi är på körningssidan
        if (window.location.href.indexOf('page=bokforehandel-runs') === -1) {
            return; // Inte på körningssidan, avsluta funktionen
        }
        
        console.log('Initierar körningssidan');
        
        // Ladda körningar direkt
        loadRuns();
        
        // Uppdatera automatiskt var 10:e sekund om det finns aktiva körningar
        let checkActiveRunsInterval = setInterval(function() {
            if ($('.runs-table tr[data-status="running"]').length > 0) {
                loadRuns(true);
            } else {
                // Avsluta intervallet om det inte finns några aktiva körningar
                clearInterval(checkActiveRunsInterval);
            }
        }, 10000);
        
        // Lägg till uppdateringsknapp för att ladda om körningarna
        $('.reload-runs-btn, #refresh-runs').click(function(e) {
            e.preventDefault();
            loadRuns();
        });
        
        // Lägg till lyssnare för filtrering
        $('#filter-run-status').change(function() {
            loadRuns();
        });
        
        // Hantera radering av körningar
        $(document).on('click', '.delete-run-btn', function(e) {
            e.preventDefault();
            
            if (confirm('Är du säker på att du vill ta bort denna körning? Detta kan inte ångras.')) {
                const runId = $(this).data('run-id');
                const $row = $(this).closest('tr');
                
                // Visa laddningsindikator
                $row.addClass('deleting');
                $(this).text('Tar bort...').prop('disabled', true);
                
                // Skicka AJAX-begäran
                ajaxRequest({
                    type: 'POST',
                    url: apiParams.ajax_url,
                    data: {
                        action: 'bokforehandel_delete_run',
                        nonce: apiParams.nonce,
                        run_id: runId
                    },
                    success: function(response) {
                        if (response.success) {
                            $row.fadeOut(400, function() {
                                $(this).remove();
                            });
                        } else {
                            $row.removeClass('deleting');
                            alert('Fel: ' + (response.data || 'Kunde inte ta bort körningen'));
                        }
                    },
                    error: function(xhr, status, error) {
                        $row.removeClass('deleting');
                        alert('Serverfel: ' + error);
                    }
                });
            }
        });
        
        // Stäng alla modalfönster när sidan laddas (säkerhetsåtgärd)
        $('.bokforehandel-modal').hide();
        
        // Hantera stängning av modalfönster
        $('.bokforehandel-modal-close').on('click', function() {
            $('#run-details-modal').hide();
        });
        
        // Ny version av show-run-details
        $(document).on('click', '.show-run-details, .view-run-details', function() {
            var runId = $(this).data('run-id');
            $('#run-details-modal').show();
            
            // Visa laddningsindikator och nollställ tidigare värden
            $('#run-details-content table td').text('Laddar...');
            $('#detail-errors').html('<p>Hämtar information...</p>');
            
            console.log('Hämtar detaljer för körning ID:', runId);
            
            $.ajax({
                url: apiParams.ajax_url,
                type: 'POST',
                data: {
                    action: 'bokforehandel_get_run_details',
                    nonce: apiParams.nonce,
                    run_id: runId
                },
                success: function(response) {
                    console.log('Körningsdetaljer svar:', response);
                    
                    if (response.success) {
                        var data = response.data;
                        $('#detail-id').text(data.id);
                        $('#detail-operation').text(getOperationName(data.operation));
                        $('#detail-date-range').text(formatDateRange(data.start_date, data.end_date, data));
                        
                        // Visa retroaktivt datumintervall om tillgängligt
                        if (data.is_retroactive && data.operations_data) {
                            $('.retroactive-details').show();
                            var retroStartDate = data.operations_data.start_date || '';
                            var retroEndDate = data.operations_data.end_date || retroStartDate;
                            $('#detail-date-range-raw').text(retroStartDate + ' - ' + retroEndDate);
                        } else {
                            $('.retroactive-details').hide();
                        }
                        
                        $('#detail-status').text(getStatusText(data.status));
                        $('#detail-items').text(data.items_processed);
                        $('#detail-runtime').text(formatRuntime(data.run_time));
                        $('#detail-date').text(formatDateTime(data.created_at));
                        
                        // Hämta och visa synkroniserade ordrar
                        loadSyncedOrders(runId);
                        
                        // Visa eventuella fel
                        if (data.error_count > 0) {
                            var errorHtml = '<p>' + data.error_count + ' fel uppstod.</p>';
                            
                            if (data.errors && data.errors.length) {
                                errorHtml += '<ul class="bokforehandel-error-list">';
                                for (var i = 0; i < data.errors.length; i++) {
                                    errorHtml += '<li>' + data.errors[i].message + '</li>';
                                }
                                errorHtml += '</ul>';
                            }
                            
                            $('#detail-errors').html(errorHtml);
                        } else {
                            $('#detail-errors').html('<p>Inga fel.</p>');
                        }
                    } else {
                        // Visa felmeddelande
                        console.error('Fel vid hämtning av körningsdetaljer:', response.data);
                        $('#detail-errors').html('<div class="notice notice-error"><p>Kunde inte hämta detaljer: ' + response.data + '</p></div>');
                        
                        // Återställ tabellen
                        $('#run-details-content table td').text('-');
                    }
                },
                error: function(xhr, status, error) {
                    // Visa detaljerat felmeddelande
                    console.error('AJAX-fel vid hämtning av körningsdetaljer:', xhr, status, error);
                    var errorMsg = getDetailedErrorMessage(xhr, status, error);
                    $('#detail-errors').html('<div class="notice notice-error"><p>AJAX-fel: ' + errorMsg + '</p></div>');
                    
                    // Återställ tabellen
                    $('#run-details-content table td').text('-');
                }
            });
        });
    }
    
    /**
     * Initiera retroaktiv synkronisering
     */
    function initRetroSync() {
        // Ändra selektorn för att matcha strukturen i HTML-filen
        console.log('Initierar retroaktiv synkronisering - startar felsökning');
        
        // Skriv ut alla formulär på sidan för felsökning
        console.log('Alla formulär på sidan:', $('form').length);
        $('form').each(function(i) {
            console.log('Formulär #' + i + ':', $(this));
        });
        
        // Kontrollera båda selectorerna
        console.log('Element med ID bokforehandel-retro-sync-form:', $('#bokforehandel-retro-sync-form').length);
        console.log('Element med class bokforehandel-retro-form:', $('.bokforehandel-retro-form').length);
        
        // Kontrollera alla element med 'retro' i klassen eller ID
        console.log('Element med retro i ID eller class:', $('[id*="retro"], [class*="retro"]').length);
        $('[id*="retro"], [class*="retro"]').each(function(i) {
            console.log('Retro element #' + i + ':', $(this).attr('id') || $(this).attr('class'));
        });
        
        // Prova med olika selektorer
        const $retroSyncForm = $('.bokforehandel-retro-form, #bokforehandel-retro-sync-form, [class*="retro-form"]').first();
        const $retroSyncBtn = $('#bokforehandel-retro-sync-btn');
        
        console.log('Hittade formulär med breda selektorer:', $retroSyncForm.length);
        
        // Om vi fortfarande inte hittar formuläret, försök hitta knappen och jobba baklänges
        if (!$retroSyncForm.length) {
            console.log('Kunde inte hitta formuläret, letar efter knappen...');
            console.log('Retro sync knapp finns:', $retroSyncBtn.length);
            
            if ($retroSyncBtn.length) {
                console.log('Hittade knappen, försöker hitta föräldraelement');
                // Prova att använda knappens närmaste föräldrar
                const $form = $retroSyncBtn.closest('div');
                console.log('Använder knappens närmaste div som formulär:', $form.length);
                
                initRetroSyncWithForm($form, $retroSyncBtn);
                return;
            }
        }
        
        if (!$retroSyncForm.length) {
            console.log('Retroaktivt synkroniseringsformulär hittades inte, alla försök misslyckades');
            // Leta efter knappen direkt som sista utväg
            const $lastResortBtn = $('button:contains("Kör retroaktiv synkronisering")');
            console.log('Knapp hittad med text-sökning:', $lastResortBtn.length);
            if ($lastResortBtn.length) {
                initRetroSyncWithForm($lastResortBtn.closest('div'), $lastResortBtn);
            }
            return;
        }
        
        initRetroSyncWithForm($retroSyncForm, $retroSyncBtn);
    }
    
    // Hjälpfunktion för att konfigurera retroaktiv synk med givet formulär
    function initRetroSyncWithForm($form, $btn) {
        console.log('Konfigurerar retroaktiv synkronisering med formulär:', $form);
        const $spinner = $form.find('.spinner');
        const $resultBox = $('#bokforehandel-retro-sync-result');
        
        // Lägg till klickhändelse på knappen direkt
        $btn.on('click', function() {
            console.log('Retro sync button clicked');
            var $button = $(this);
            var $spinner = $button.next('.spinner');
            var startDate = $('#retro_sync_start_date').val();
            var endDate = $('#retro_sync_end_date').val();
            var syncType = $('#retro_sync_type').val();
            
            // FELSÖKNING: Dumpa params för att verifiera att de finns
            console.log('FELSÖKNING: apiParams:', apiParams);
            console.log('FELSÖKNING: bokforehandel:', typeof bokforehandel !== 'undefined' ? bokforehandel : 'undefined');
            console.log('FELSÖKNING: ajaxurl:', typeof ajaxurl !== 'undefined' ? ajaxurl : 'undefined');
            
            if (!startDate) {
                alert('Vänligen ange ett startdatum.');
                return;
            }
            
            if (!endDate) {
                alert('Vänligen ange ett slutdatum.');
                return;
            }
            
            // FELSÖKNING: Kontrollera datum
            console.log('FELSÖKNING: Datum före omformning:', {startDate, endDate});
            
            // Konvertera från YYYYMMDD till YYYY-MM-DD
            function formatDateForAPI(dateStr) {
                // Om det redan är i formatet YYYY-MM-DD, returnera det
                if (/^\d{4}-\d{2}-\d{2}$/.test(dateStr)) {
                    return dateStr;
                }
                
                // Om det är YYYYMMDD (8 siffror), formatera det som YYYY-MM-DD
                if (/^\d{8}$/.test(dateStr)) {
                    return dateStr.substring(0, 4) + '-' + dateStr.substring(4, 6) + '-' + dateStr.substring(6, 8);
                }
                
                // Annars returnera originalet
                return dateStr;
            }
            
            // Säkerställ att datumen är i rätt format för API:et (YYYY-MM-DD)
            startDate = formatDateForAPI(startDate);
            endDate = formatDateForAPI(endDate);
            
            console.log('Retroaktiv synkronisering med datum:', startDate, endDate);
            console.log('FELSÖKNING: Datum efter omformning:', {startDate, endDate});
            
            // Visa visuell indikation på att synkroniseringen startar
            $button.text('Kör...').prop('disabled', true);
            $spinner.addClass('is-active');
            $('#bokforehandel-retro-sync-result').html('<div class="notice notice-info"><p>Startar retroaktiv synkronisering...</p><div class="spinner is-active" style="float:none;"></div></div>');
            
            // Kontrollera att vi har en AJAX-URL
            const ajaxUrl = apiParams.ajax_url || (typeof ajaxurl !== 'undefined' ? ajaxurl : null);
            if (!ajaxUrl) {
                console.error('FELSÖKNING: AJAX URL saknas! Kan inte fortsätta.');
                alert('Ett fel uppstod: AJAX URL saknas. Ladda om sidan och försök igen.');
                $button.text('Kör retroaktiv synkronisering').prop('disabled', false);
                $spinner.removeClass('is-active');
                return;
            }
            
            // Kontrollera att vi har en nonce
            const nonce = apiParams.nonce || '';
            if (!nonce) {
                console.warn('FELSÖKNING: Nonce saknas, vilket kan leda till säkerhetsfel.');
            }
            
            // Förbered data för AJAX-anropet med detaljerad loggning
            const ajaxData = {
                action: 'bokforehandel_run_retro_sync',
                nonce: nonce,
                start_date: startDate,
                end_date: endDate,
                sync_type: syncType,
                is_retroactive: true
            };
            
            console.log('FELSÖKNING: Skickar AJAX-anrop till:', ajaxUrl);
            console.log('FELSÖKNING: Med data:', ajaxData);
            
            // Utför AJAX-anropet
            $.ajax({
                url: ajaxUrl,
                type: 'POST',
                data: ajaxData,
                beforeSend: function() {
                    console.log('FELSÖKNING: AJAX-anrop skickas nu...');
                },
                success: function(response) {
                    console.log('FELSÖKNING: AJAX-anrop lyckades, svar:', response);
                    if (response.success) {
                        $('#bokforehandel-retro-sync-result').html('<div class="notice notice-success">' +
                            '<p><strong>Synkroniseringen har startats!</strong></p>' +
                            '<p>' + response.data.message + '</p>' +
                            '<p>Synkroniseringen körs nu i bakgrunden. Du kan följa förloppet på "Körningar"-fliken.</p>' +
                            '</div>');
                    } else {
                        console.error('FELSÖKNING: AJAX-anrop returnerade fel:', response);
                        let felmeddelande = 'Ett fel uppstod.';
                        if (response.data && response.data.message) {
                            felmeddelande = response.data.message;
                        }
                        $('#bokforehandel-retro-sync-result').html('<div class="notice notice-error">' +
                            '<p><strong>Fel:</strong> ' + felmeddelande + '</p>' +
                            '<p>Kontrollera att datumen är i rätt format (YYYYMMDD) och att du har behörighet att köra synkroniseringen.</p>' +
                            '</div>');
                    }
                    $button.text('Kör retroaktiv synkronisering').prop('disabled', false);
                    $spinner.removeClass('is-active');
                },
                error: function(xhr, status, error) {
                    console.error('FELSÖKNING: AJAX-fel:', {xhr: xhr, status: status, error: error});
                    console.error('FELSÖKNING: AJAX responseText:', xhr.responseText);
                    
                    // Försök hantera JSON-svar även på error
                    let errorMessage = error || 'Okänt fel';
                    try {
                        if (xhr.responseText) {
                            const errorResponse = JSON.parse(xhr.responseText);
                            if (errorResponse && errorResponse.data && errorResponse.data.message) {
                                errorMessage = errorResponse.data.message;
                            }
                        }
                    } catch (e) {
                        console.error('FELSÖKNING: Kunde inte tolka felsvaret som JSON:', e);
                    }
                    
                    $('#bokforehandel-retro-sync-result').html('<div class="notice notice-error">' +
                        '<p><strong>Serverfel:</strong> ' + errorMessage + '</p>' +
                        '<p>Kontrollera din WordPress-konfiguration och nätverksanslutning.</p>' +
                        '<p><em>Teknisk information:</em> Status: ' + status + ', Fel: ' + error + '</p>' +
                        '</div>');
                    $button.text('Kör retroaktiv synkronisering').prop('disabled', false);
                    $spinner.removeClass('is-active');
                }
            });
        });
    }
    
    /**
     * Initiera manuell synkronisering
     */
    function initManualSync() {
        // Kontrollera om elementet finns på sidan
        if (!$('#manual-sync-button').length) {
            return;
        }
        
        $('#manual-sync-button').on('click', function(e) {
            e.preventDefault();
            
            var $button = $(this);
            var originalText = $button.text();
            var $statusElement = $('#manual-sync-status');
            
            // Inaktivera knappen och visa status
            $button.attr('disabled', true).text('Synkroniserar...');
            $statusElement.html('<p class="syncing">Synkroniserar föregående dags ordrar...</p>');
            
            // Gör AJAX-anropet med ajaxRequest-hjälpfunktionen
            ajaxRequest({
                type: 'POST',
                url: apiParams.ajax_url,
                data: {
                    action: 'bokforehandel_manual_sync',
                    nonce: apiParams.nonce
                },
                success: function(response) {
                    console.log('Manual sync response:', response);
                    
                    if (response.success) {
                        var runId = response.data.run_id;
                        $statusElement.html('<p class="success">Synkronisering startad! Följ förloppet på <a href="?page=bokforehandel-runs">Körningar</a>-sidan.</p>');
                        
                        // Kontrollera status för denna körning regelbundet
                        if (runId) {
                            var statusCheckInterval = setInterval(function() {
                                checkRunStatus(runId, function(status, progress) {
                                    if (status === 'completed') {
                                        clearInterval(statusCheckInterval);
                                        $statusElement.html('<p class="success">Synkronisering slutförd! Se detaljerna på <a href="?page=bokforehandel-runs">Körningar</a>-sidan.</p>');
                                    } else if (status === 'failed') {
                                        clearInterval(statusCheckInterval);
                                        $statusElement.html('<p class="error">Synkronisering misslyckades. Se detaljerna på <a href="?page=bokforehandel-runs">Körningar</a>-sidan.</p>');
                                    } else if (status === 'error') {
                                        clearInterval(statusCheckInterval);
                                        $statusElement.html('<p class="error">Ett fel inträffade under synkroniseringen. Se detaljerna på <a href="?page=bokforehandel-runs">Körningar</a>-sidan.</p>');
                                    }
                                });
                            }, 5000); // Kontrollera var 5:e sekund
                        }
                    } else {
                        let errorMsg = 'Kunde inte starta synkronisering';
                        
                        // Hantera olika typer av felsvar
                        if (response.data) {
                            if (typeof response.data === 'object') {
                                errorMsg = response.data.message || JSON.stringify(response.data);
                            } else {
                                errorMsg = response.data;
                            }
                        }
                        
                        $statusElement.html('<p class="error">Fel: ' + errorMsg + '</p>');
                        console.error('Manual sync error:', response);
                    }
                    
                    // Återställ knappen efter 3 sekunder
                    setTimeout(function() {
                        $button.attr('disabled', false).text(originalText);
                    }, 3000);
                },
                error: function(xhr, status, error) {
                    let errorMsg = 'Kunde inte genomföra synkronisering';
                    
                    try {
                        const response = JSON.parse(xhr.responseText);
                        if (response && response.data) {
                            if (typeof response.data === 'object') {
                                errorMsg = response.data.message || JSON.stringify(response.data);
                            } else {
                                errorMsg = response.data;
                            }
                        }
                    } catch (e) {
                        errorMsg = error || status || 'Okänt fel';
                    }
                    
                    $statusElement.html('<p class="error">Fel: ' + errorMsg + '</p>');
                    console.error('AJAX error in manual sync:', {xhr, status, error});
                    
                    // Återställ knappen efter 3 sekunder
                    setTimeout(function() {
                        $button.attr('disabled', false).text(originalText);
                    }, 3000);
                }
            });
        });
    }
    
    // ========================
    // Hjälpfunktioner
    // ========================
    
    /**
     * Förbättrad AJAX-begäran med cachning och återförsök
     */
    function ajaxRequest(options) {
        const cacheKey = options.cacheKey || null;
        const maxRetries = options.maxRetries || 2;
        let currentRetry = 0;
        
        // Om vi har en cache-nyckel och cachen är giltig, använd den
        if (cacheKey && apiResponseCache[cacheKey] && 
            (Date.now() - apiResponseCache[cacheKey].timestamp) < cacheExpiry) {
            
            if (options.success) {
                options.success(apiResponseCache[cacheKey].data);
            }
            return;
        }
        
        // Funktionen för att göra den faktiska AJAX-begäran
        function makeRequest() {
            // Logga begäran för debugging
            if (options.logRequest) {
                console.log('AJAX-begäran:', {
                    type: options.type || 'GET',
                    url: options.url,
                    data: options.data
                });
            }
            
            // Skapa ett timeout-löfte
            const timeoutPromise = new Promise((_, reject) => {
                setTimeout(() => reject(new Error('Timeout')), 30000); // 30 sekunder timeout
            });
            
            // Skapa AJAX-löftet
            const ajaxPromise = new Promise((resolve, reject) => {
                var ajaxOptions = {
                    type: options.type || 'GET',
                    url: options.url,
                    data: options.data,
                    processData: options.processData !== undefined ? options.processData : true,
                    contentType: options.contentType !== undefined ? options.contentType : 'application/x-www-form-urlencoded; charset=UTF-8',
                    dataType: options.dataType || 'json',
                    headers: options.headers || {},
                    success: function(response) {
                        resolve(response);
                    },
                    error: function(xhr, status, error) {
                        reject({ xhr, status, error });
                    }
                };
                
                // Logga headers för debug
                if (options.logRequest && options.headers) {
                    console.log('AJAX headers:', options.headers);
                }
                
                jQuery.ajax(ajaxOptions);
            });
            
            // Använd det löfte som löses först (ajax eller timeout)
            Promise.race([ajaxPromise, timeoutPromise])
                .then(response => {
                    // Spara i cache om vi har en cache-nyckel
                    if (cacheKey) {
                        apiResponseCache[cacheKey] = {
                            data: response,
                            timestamp: Date.now()
                        };
                    }
                    
                    if (options.success) {
                        options.success(response);
                    }
                })
                .catch(error => {
                    // Om det är ett timeout-fel eller nätverksfel
                    const isTimeoutOrNetworkError = 
                        (error.message === 'Timeout') || 
                        (error.status === 'timeout') || 
                        (error.xhr && error.xhr.status === 0) || 
                        (error.xhr && error.xhr.status >= 500);
                    
                    // Försök igen vid timeout, nätverksfel eller serverfel
                    if (currentRetry < maxRetries && isTimeoutOrNetworkError) {
                        console.warn(`AJAX-begäran misslyckades, försöker igen (${currentRetry + 1}/${maxRetries})`, error);
                        currentRetry++;
                        setTimeout(makeRequest, 1000 * currentRetry); // Vänta längre för varje försök
                    } else if (options.error) {
                        // Om vi har ett Error-objekt
                        if (error.message === 'Timeout') {
                            options.error({ 
                                status: 'timeout', 
                                error: 'Timeout efter 30 sekunder'
                            });
                        } else {
                            // Om vi har ett objekt från jQuery.ajax's error callback
                            options.error(error.xhr, error.status, error.error);
                        }
                    }
                });
        }
        
        // Starta begäran
        makeRequest();
    }
    
    /**
     * Laddar körningar med cachning
     */
    function loadRuns(refresh = false) {
        // Visa laddningsindikator
        $('.bokforehandel-runs-loading').show();
        $('.bokforehandel-runs-table').hide();
        $('#no-runs').hide();
        
        // Skicka AJAX-begäran med cachning
        ajaxRequest({
            type: 'POST',
            url: apiParams.ajax_url,
            data: {
                action: 'bokforehandel_get_runs',
                nonce: apiParams.nonce
            },
            headers: {
                'X-User-ID': apiParams.user_id
            },
            cacheKey: 'runs', // Använd cachning
            success: function(response) {
                // Dölj laddningsindikator
                $('.bokforehandel-runs-loading').hide();
                
                if (response.success) {
                    var runs = response.data;
                    
                    if (runs && runs.length > 0) {
                        // Rensa tabellen
                        $('#runs-list').empty();
                        
                        // Fyll i tabellen
                        for (var i = 0; i < runs.length; i++) {
                            var run = runs[i];
                            
                            var row = $('<tr></tr>');
                            row.append('<td>' + run.id + '</td>');
                            row.append('<td>' + getOperationName(run.operation) + '</td>');
                            row.append('<td>' + formatDateRange(run.start_date, run.end_date, run) + '</td>');
                            row.append('<td>' + getStatusLabel(run.status) + '</td>');
                            row.append('<td>' + run.items_processed + '</td>');
                            row.append('<td>' + formatRuntime(run.run_time) + '</td>');
                            row.append('<td>' + formatDateTime(run.created_at) + '</td>');
                            row.append('<td><button class="button view-run-details" data-run-id="' + run.id + '">Visa detaljer</button></td>');
                            
                            $('#runs-list').append(row);
                        }
                        
                        // Visa tabellen
                        $('.bokforehandel-runs-table').show();
                    } else {
                        // Visa meddelande om inga körningar
                        $('#no-runs').show();
                    }
                } else {
                    // Visa felmeddelande
                    showNotice('Fel: ' + response.data, 'error');
                }
            },
            error: function(xhr, status, error) {
                $('.bokforehandel-runs-loading').hide();
                showNotice('Serverfel vid hämtning av körningar: ' + error, 'error');
            }
        });
    }
    
    /**
     * Laddar och visar synkroniserade ordrar för en specifik körning
     */
    function loadSyncedOrders(runId) {
        const $ordersContainer = $('#detail-synced-orders');
        const $loadingIndicator = $ordersContainer.find('.orders-loading');
        const $ordersTable = $ordersContainer.find('.orders-table');
        const $noOrders = $ordersContainer.find('.no-orders');
        const $ordersList = $('#synced-orders-list');
        
        console.log('Hämtar synkroniserade ordrar för körning:', runId);
        
        // Visa laddar-indikator och dölj annat innehåll
        $loadingIndicator.show();
        $ordersTable.hide();
        $noOrders.hide();
        
        // Töm tidigare orderdata
        $ordersList.empty();
        
        // Hämta synkroniserade ordrar via AJAX
        ajaxRequest({
            type: 'POST',
            url: apiParams.ajax_url,
            data: {
                action: 'bokforehandel_get_synced_orders',
                nonce: apiParams.nonce,
                run_id: runId
            },
            success: function(response) {
                $loadingIndicator.hide();
                console.log('Svar från synkroniserade ordrar:', response);
                
                if (!response || !response.success) {
                    const errorMsg = response && response.data ? response.data : 'Okänt fel';
                    console.error('Fel vid hämtning av synkroniserade ordrar:', errorMsg);
                    $noOrders.html('<p>Fel vid hämtning av ordrar: ' + errorMsg + '</p>').show();
                    return;
                }
                
                // Säkerställ att vi har orders-fältet
                if (!response.data || !response.data.orders) {
                    console.warn('Svaret saknar orders-fältet:', response.data);
                    $noOrders.html('<p>Inga synkroniserade ordrar hittades (tomt API-svar)</p>').show();
                    return;
                }
                
                const orders = response.data.orders;
                
                // Kontrollera att orders är en array
                if (!Array.isArray(orders)) {
                    console.warn('orders är inte en array:', orders);
                    $noOrders.html('<p>Fel format på synkroniserade ordrar</p>').show();
                    return;
                }
                
                if (orders.length === 0) {
                    console.log('Inga synkroniserade ordrar hittades');
                    $noOrders.html('<p>Inga synkroniserade ordrar hittades</p>').show();
                    return;
                }
                
                console.log('Hittade ' + orders.length + ' synkroniserade ordrar');
                
                // Bygga tabellrader för varje order med förbättrad felhantering
                orders.forEach(function(order, index) {
                    try {
                        // Säkerställ att order är ett objekt
                        if (!order || typeof order !== 'object') {
                            console.warn('Order på index ' + index + ' är inte ett objekt:', order);
                            return;
                        }
                        
                        const rowHtml = `
                            <tr>
                                <td>${order.order_id || '-'}</td>
                                <td>${formatDateTime(order.date) || '-'}</td>
                                <td>${formatCurrency(order.amount) || '-'}</td>
                                <td>${order.payment_method || '-'}</td>
                                <td>${order.status || '-'}</td>
                                <td>
                                    <a href="${order.admin_url || '#'}" target="_blank" class="button button-small">Visa order</a>
                                </td>
                            </tr>
                        `;
                        $ordersList.append(rowHtml);
                    } catch (e) {
                        console.error('Fel vid rendering av order ' + index + ':', e, order);
                    }
                });
                
                // Visa tabellen endast om det finns rader
                if ($ordersList.children().length > 0) {
                    $ordersTable.show();
                } else {
                    $noOrders.html('<p>Kunde inte visa synkroniserade ordrar</p>').show();
                }
            },
            error: function(xhr, status, error) {
                $loadingIndicator.hide();
                console.error('AJAX-fel vid hämtning av synkroniserade ordrar:', xhr, status, error);
                
                let errorMessage = 'Serverfel vid hämtning av ordrar';
                
                // Försök hämta mer detaljerad felinformation
                if (xhr && xhr.responseJSON && xhr.responseJSON.data) {
                    errorMessage += ': ' + xhr.responseJSON.data;
                } else if (error) {
                    errorMessage += ': ' + error;
                }
                
                $noOrders.html('<p>' + errorMessage + '</p>').show();
            }
        });
    }
    
    /**
     * Hjälpfunktion för att formatera valutabelopp
     */
    function formatCurrency(amount) {
        if (!amount && amount !== 0) return '-';
        return parseFloat(amount).toFixed(2) + ' kr';
    }
    
    /**
     * Kontrollerar status för en pågående körning
     */
    function checkRunStatus(runId, callback) {
        console.log('Kontrollerar status för körning:', runId);
        
        const $progressBar = $('.integration-progress');
        const $progressText = $('.integration-progress-text');
        const $resultContainer = $('.integration-results');
        const $retroSyncBtn = $('#bokforehandel-retro-sync-btn');
        
        if (!runId) {
            console.error('Inget run_id angavs för statuskontroll');
            $progressText.text('Fel: Inget körnings-ID tillgängligt');
            $retroSyncBtn.prop('disabled', false);
            return;
        }
        
        ajaxRequest({
            type: 'POST',
            url: apiParams.ajax_url,
            data: {
                action: 'bokforehandel_check_status',
                nonce: apiParams.nonce,
                run_id: runId
            },
            success: function(response) {
                console.log('Status kontrollsvar:', response);
                
                if (!response.success) {
                    let errorMsg = response.data && response.data.message 
                        ? response.data.message 
                        : 'Ett fel uppstod vid kontroll av körningsstatus';
                    $progressText.text('Fel: ' + errorMsg);
                    $retroSyncBtn.prop('disabled', false);
                    
                    // Visa felmeddelande i resultatlådan
                    $resultContainer.html('<div class="notice notice-error"><p>Fel vid statuskontroll: ' + errorMsg + '</p></div>').show();
                    return;
                }
                
                const data = response.data;
                const status = data.status;
                let progress = 0;
                
                // Uppdatera processtext och förloppsindikator baserat på status
                if (status === 'running') {
                    progress = data.progress || 50;
                    $progressText.text('Bearbetar... ' + (data.processed_items || 0) + ' objekt hanterade');
                    
                    // Schemalägg en ny kontroll om 2 sekunder
                    setTimeout(function() {
                        checkRunStatus(runId, callback);
                    }, 2000);
                } else if (status === 'completed') {
                    progress = 100;
                    $progressText.text('Klar! ' + (data.processed_items || 0) + ' objekt har synkroniserats.');
                    
                    // Visa resultaten
                    let resultsHtml = '<div class="notice notice-success"><p>Synkronisering slutförd!</p>';
                    
                    if (data.details) {
                        resultsHtml += '<ul>';
                        if (data.details.orders) resultsHtml += '<li>Ordrar: ' + data.details.orders + '</li>';
                        if (data.details.payments) resultsHtml += '<li>Betalningar: ' + data.details.payments + '</li>';
                        if (data.details.refunds) resultsHtml += '<li>Återbetalningar: ' + data.details.refunds + '</li>';
                        resultsHtml += '</ul>';
                    }
                    
                    resultsHtml += '</div>';
                    $resultContainer.html(resultsHtml).show();
                } else if (status === 'failed') {
                    progress = 100;
                    $progressText.text('Misslyckades: ' + (data.message || 'Okänt fel'));
                    
                    // Visa feldetaljer
                    let errorHtml = '<div class="notice notice-error"><p>Synkronisering misslyckades</p>';
                    
                    if (data.errors && data.errors.length > 0) {
                        errorHtml += '<ul class="error-list">';
                        data.errors.forEach(function(error) {
                            errorHtml += '<li>' + error + '</li>';
                        });
                        errorHtml += '</ul>';
                    }
                    
                    errorHtml += '</div>';
                    $resultContainer.html(errorHtml).show();
                }
                
                // Uppdatera förloppsindikatorn
                $progressBar.css('width', progress + '%');
                
                if (callback) {
                    callback(status, progress);
                }
            },
            error: function(xhr, status, error) {
                console.error('Fel vid kontroll av status:', xhr, status, error);
                
                // Visa felmeddelande
                let errorMsg = 'Kunde inte kontrollera status';
                
                try {
                    const response = JSON.parse(xhr.responseText);
                    if (response && response.data) {
                        if (typeof response.data === 'object') {
                            errorMsg = response.data.message || JSON.stringify(response.data);
                        } else {
                            errorMsg = response.data;
                        }
                    }
                } catch (e) {
                    errorMsg = error || status || 'Okänt fel';
                }
                
                // Visa felmeddelande i UI
                if ($('#run-' + runId + '-details').length) {
                    $('#run-' + runId + '-details .status-message').html('<span class="error">' + errorMsg + '</span>');
                }
                
                if (callback) {
                    callback('error', 0);
                }
            }
        });
    }
    
    /**
     * Förbättrad funktion för att visa meddelanden
     */
    window.showNotice = function(message, type) {
        type = type || 'success';
        
        // Skapa en container för meddelanden om den inte finns
        var $noticeContainer = $('.notice-container');
        if ($noticeContainer.length === 0) {
            $noticeContainer = $('<div class="notice-container"></div>');
            $('.bokforehandel-admin-header').after($noticeContainer);
        }
        
        // Skapa starkare meddelanden med tydligare information
        var noticePrefix = '';
        if (type === 'error') {
            noticePrefix = '<strong>Fel:</strong> ';
        } else if (type === 'success') {
            noticePrefix = '<strong>Klart:</strong> ';
        }
        
        // Skapa meddelandet med dismiss-knapp
        var $notice = $('<div class="notice notice-' + type + ' bokforehandel-notice is-dismissible"><p>' + 
                        noticePrefix + message + '</p>' +
                        '<button type="button" class="notice-dismiss"><span class="screen-reader-text">Stäng detta meddelande.</span></button>' +
                        '</div>');
        
        // Lägg till och markera meddelandet
        $noticeContainer.append($notice);
        $notice.addClass('bokforehandel-notice-highlight');
        
        // Scrolla till meddelandet om det inte syns
        $('html, body').animate({
            scrollTop: $notice.offset().top - 100
        }, 500);
        
        // Konfigurera dismiss-knappen
        $notice.find('.notice-dismiss').on('click', function() {
            $notice.fadeOut(300, function() {
                $(this).remove();
            });
        });
        
        // Automatisk fade-out endast för framgångsmeddelanden
        if (type === 'success') {
            setTimeout(function() {
                $notice.fadeOut(500, function() {
                    $(this).remove();
                });
            }, 30000);
        }
        
        return $notice;
    };
    
    /**
     * Hjälpfunktion för att få fliknamn
     */
    function getTabName(tab) {
        return '';
    }
    
    /**
     * Formaterar statustext för körningar
     */
    function getStatusText(status) {
        switch (status) {
            case 'running': return 'Pågår';
            case 'completed': return 'Slutförd';
            case 'failed': return 'Misslyckad';
            default: return status;
        }
    }
    
    /**
     * Hjälpfunktion för att formatera körtid
     */
    function formatRuntime(seconds) {
        if (seconds < 60) {
            return seconds + ' sekunder';
        } else {
            var minutes = Math.floor(seconds / 60);
            var remainingSeconds = seconds % 60;
            return minutes + ' minuter ' + remainingSeconds + ' sekunder';
        }
    }
    
    /**
     * Formaterar en status som en etikett
     */
    function getStatusLabel(status) {
        switch (status) {
            case 'running': return '<span class="bokforehandel-status running">Pågår</span>';
            case 'completed': return '<span class="bokforehandel-status completed">Slutförd</span>';
            case 'failed': return '<span class="bokforehandel-status failed">Misslyckad</span>';
            default: return status;
        }
    }
    
    /**
     * Returnerar namn på operationer
     */
    function getOperationName(operation) {
        switch (operation) {
            case 'all': return 'Alla';
            case 'woo': return 'WooCommerce';
            case 'klarna': return 'Klarna';
            case 'stripe': return 'Stripe';
            case 'wooklarna': return 'WooCommerce + Klarna';
            case 'wooswish': return 'WooCommerce (Swish)';
            case 'woostripe': return 'WooCommerce (Stripe)';
            case 'refund': return 'Återbetalningar';
            default: return operation;
        }
    }
    
    /**
     * Formaterar ett datumintervall
     */
    function formatDateRange(start, end, run) {
        // Om det är en retroaktiv synkronisering och vi har operations_data med date_range
        if (run && run.is_retroactive && run.operations_data && run.operations_data.date_range) {
            // Om vi har saved_start_date och saved_end_date från API-svaret
            if (run.saved_start_date && run.saved_end_date) {
                var startDate = new Date(run.saved_start_date);
                var endDate = new Date(run.saved_end_date);
                
                if (!isNaN(startDate.getTime()) && !isNaN(endDate.getTime())) {
                    var startFormatted = startDate.toLocaleDateString('sv-SE');
                    var endFormatted = endDate.toLocaleDateString('sv-SE');
                    return startFormatted + ' - ' + endFormatted + ' (retroaktiv)';
                }
            }
            
            // Visa rått datumintervall om det finns
            var dateRange = run.operations_data.date_range;
            if (dateRange && dateRange.indexOf('-') > 0) {
                var dates = dateRange.split('-');
                if (dates.length === 2) {
                    // Formatera YYYYMMDD till YYYY-MM-DD för läsbarhet
                    var startStr = dates[0];
                    var endStr = dates[1];
                    
                    if (startStr.length === 8 && endStr.length === 8) {
                        var formattedStart = startStr.substring(0, 4) + '-' + startStr.substring(4, 6) + '-' + startStr.substring(6, 8);
                        var formattedEnd = endStr.substring(0, 4) + '-' + endStr.substring(4, 6) + '-' + endStr.substring(6, 8);
                        return formattedStart + ' - ' + formattedEnd + ' (retroaktiv)';
                    }
                    
                    return dateRange + ' (retroaktiv)';
                }
            }
        }
        
        // Standardformatering om det inte är en retroaktiv synkronisering eller om vi saknar operations_data
        if (!start || !end) return 'N/A';
        
        var startDate = new Date(start);
        var endDate = new Date(end);
        
        if (isNaN(startDate.getTime()) || isNaN(endDate.getTime())) {
            return 'Ogiltigt datum';
        }
        
        // Formatera datum i svensk format
        var startFormatted = startDate.toLocaleDateString('sv-SE');
        var endFormatted = endDate.toLocaleDateString('sv-SE');
        
        return startFormatted + (startFormatted !== endFormatted ? ' - ' + endFormatted : '');
    }
    
    /**
     * Formaterar ett datum med tid
     */
    function formatDateTime(dateTimeStr) {
        if (!dateTimeStr) return 'N/A';
        
        var date = new Date(dateTimeStr);
        
        if (isNaN(date.getTime())) {
            return 'Ogiltigt datum';
        }
        
        // Formatera datum och tid i svensk format
        return date.toLocaleString('sv-SE');
    }

    // Funktion för att kontrollera integrationsstatus
    function checkIntegrationStatus() {
        var $statusEl = $('#bl-integration-status');
        $statusEl.html('<span class="checking">Kontrollerar...</span>');

        $.ajax({
            url: apiParams.ajax_url,
            type: 'POST',
            data: {
                action: 'bokforehandel_check_status',
                nonce: apiParams.nonce
            },
            success: function(response) {
                if (response.success) {
                    var status = response.data.integration_status;
                    if (status === 'on') {
                        $statusEl.html('<span class="status-on">Aktiverad</span>');
                        // Dölj alltid hjälprutan när integrationen är aktiverad
                        $('#bl-activation-help').hide();
                    } else if (status === 'off') {
                        $statusEl.html(
                            '<div class="status-box">' +
                            '<span class="status-off">Ej aktiverad</span>' +
                            '<a href="#" class="bl-help-link">Hur aktiverar jag?</a>' +
                            '</div>'
                        );
                        
                        // Uppdatera hjälptexten för aktivering
                        var $helpContent = $('#bl-activation-help');
                        if ($helpContent.length === 0) {
                            // Om hjälprutan inte finns, skapa den
                            $helpContent = $('<div id="bl-activation-help" class="bl-help-content" style="display:none;"></div>');
                            $statusEl.after($helpContent);
                        }
                        
                        // Uppdatera innehållet i hjälprutan med tydligare styling
                        $helpContent.html(`
                            <div class="activation-help-inner">
                                <h3>Aktivera integration med Bjorn Lunden</h3>
                                <p>För att aktivera integrationen behöver du:</p>
                                <ol>
                                    <li>Kontrollera att ditt organisationsnummer och BL User Key är korrekt ifyllda i inställningarna.</li>
                                    <li>Logga in på <a href="https://bjornlunden.se" target="_blank">bjornlunden.se</a> för att aktivera integrationen för din butik.</li>
                                    <li>När integrationen har aktiverats från Bjorn Lunden, uppdatera denna sida för att se aktuell status.</li>
                                </ol>
                                <p class="help-footer">Om du har frågor om aktiveringen, kontakta Bjorn Lundens support.</p>
                            </div>
                        `);
                        
                        // Visa hjälpinnehållet med snygg animation
                        if ($('.bl-help-link').hasClass('active')) {
                            $helpContent.show();
                        } else {
                            $helpContent.hide();
                        }
                    } else {
                        $statusEl.html('<span class="status-error">Okänd status: ' + status + '</span>');
                        $('#bl-activation-help').hide();
                    }
                } else {
                    $statusEl.html('<span class="status-error">' + response.data.message + '</span>');
                    $('#bl-activation-help').hide();
                }
            },
            error: function(xhr, status, error) {
                var errorMsg = xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message 
                    ? xhr.responseJSON.data.message 
                    : 'Kunde inte kontrollera status: ' + error;
                $statusEl.html('<span class="status-error">' + errorMsg + '</span>');
                $('#bl-activation-help').hide();
            }
        });
    }
    
    // Förbättrad hantering av hjälplänken för att visa/dölja hjälpinnehållet
    $(document).on('click', '.bl-help-link', function(e) {
        e.preventDefault();
        $(this).toggleClass('active');
        $('#bl-activation-help').slideToggle(300);
    });

    // Uppdatera status efter att inställningar sparats
    $(document).on('bokforehandel_settings_saved', function() {
        setTimeout(checkIntegrationStatus, 1000);
    });

    /**
     * Testar nätverksanslutningen till API:et
     */
    function testNetworkConnection() {
        return new Promise((resolve, reject) => {
            // Kontrollera att API-parametrar finns
            if (!apiParams || !apiParams.ajax_url || !apiParams.nonce) {
                console.error('API-parametrar saknas vid nätverkstest', apiParams);
                reject('Konfigurationsfel: API-parametrar saknas');
                return;
            }
            
            console.log('Testar nätverksanslutning till', apiParams.ajax_url);
            
            // Använd standard jQuery AJAX för nätverkstestet istället för ajaxRequest
            // Detta ger oss mer kontroll och undviker rekursion om ajaxRequest har problem
            $.ajax({
                type: 'POST',
                url: apiParams.ajax_url,
                data: {
                    action: 'bokforehandel_test_network',
                    nonce: apiParams.nonce
                },
                timeout: 15000, // 15 sekunder timeout
                success: function(response) {
                    console.log('Nätverkstest resultat:', response);
                    
                    if (response && response.success) {
                        resolve(true);
                    } else {
                        var errorMsg = (response && response.data) ? response.data : 'Nätverkstest misslyckades';
                        console.warn('Nätverkstest misslyckades:', errorMsg);
                        reject(errorMsg);
                    }
                },
                error: function(xhr, status, error) {
                    var detailedError = error || 'Kunde inte nå API:et';
                    
                    // Lägg till mer specifika fel baserat på status
                    if (status === 'timeout') {
                        detailedError = 'Tidsgräns uppnådd vid anslutning till API:et';
                    } else if (xhr && xhr.status === 0) {
                        detailedError = 'Kunde inte ansluta till API:et. Kontrollera din internetanslutning.';
                    } else if (xhr && xhr.status === 403) {
                        detailedError = 'Tillträde nekad (403). Sessionen kan ha gått ut.';
                    } else if (xhr && xhr.status === 404) {
                        detailedError = 'API-endpunkten kunde inte hittas (404).';
                    } else if (xhr && xhr.status >= 500) {
                        detailedError = 'Serverfel (' + xhr.status + '). Vänligen försök igen senare.';
                    }
                    
                    console.error('Nätverkstest AJAX-fel:', {
                        error: detailedError,
                        status: status,
                        xhr_status: xhr ? xhr.status : 'N/A',
                        responseText: xhr ? xhr.responseText : 'Inget svar'
                    });
                    
                    reject(detailedError);
                }
            });
        });
    }

    /**
     * Hantera klick på "Testa anslutning"-knappen
     */
    function testConnectionClicked() {
        // Visa laddningsindikator
        $('#test-connection').prop('disabled', true).text('Testar...');
        $('#bl-integration-status').html('<span class="checking">Testar anslutning...</span>');
        
        // Testa anslutning med mer detaljerade steg
        console.log('Testar anslutning manuellt...');
        
        // Steg 1: Kontrollera API-parametrar
        if (!apiParams || !apiParams.ajax_url) {
            console.error('API-parametrar saknas vid nätverkstest', apiParams);
            showConnectionError('API-URL saknas. Spara inställningarna först.');
            return;
        }
        
        if (!apiParams.nonce) {
            console.error('Säkerhetsnonce saknas vid nätverkstest', apiParams);
            showConnectionError('Säkerhetsnonce saknas. Ladda om sidan och försök igen.');
            return;
        }
        
        if (!apiParams.user_id) {
            console.error('Användar-ID saknas vid nätverkstest', apiParams);
            showConnectionError('Användar-ID saknas. Du måste vara inloggad och ha sparat dina inställningar.');
            return;
        }
        
        // Steg 2: Ping-test mot WordPress
        ajaxRequest({
            type: 'POST',
            url: apiParams.ajax_url,
            data: {
                action: 'bokforehandel_ping',
                nonce: apiParams.nonce
            },
            success: function(pingResponse) {
                console.log('Ping-test resultat:', pingResponse);
                
                if (pingResponse && pingResponse.success) {
                    // Steg 3: Testa faktisk BL API-anslutning
                    $('#bl-integration-status').html('<span class="checking">WordPress OK. Testar API...</span>');
                    
                    ajaxRequest({
                        type: 'POST',
                        url: apiParams.ajax_url,
                        data: {
                            action: 'bokforehandel_test_bl_api',
                            nonce: apiParams.nonce
                        },
                        success: function(apiResponse) {
                            console.log('BL API-test resultat:', apiResponse);
                            
                            if (apiResponse && apiResponse.success) {
                                // Visa framgång
                                $('#bl-integration-status').html('<span class="status-on">Anslutningstest lyckades!</span>');
                                showNotice('Anslutningstest lyckades! API-kommunikationen fungerar.', 'success');
                                
                                // Uppdatera status efter kort fördröjning
                                setTimeout(checkIntegrationStatus, 3000);
                            } else {
                                var errorMsg = (apiResponse && apiResponse.data) ? apiResponse.data : 'Okänt API-fel';
                                showConnectionError('BL API-fel: ' + errorMsg);
                                // Återställ knapp
                                $('#test-connection').prop('disabled', false).text('Testa anslutning');
                            }
                        },
                        error: function(xhr, status, error) {
                            var errorMsg = getDetailedErrorMessage(xhr, status, error);
                            showConnectionError('BL API-fel: ' + errorMsg);
                            // Återställ knapp
                            $('#test-connection').prop('disabled', false).text('Testa anslutning');
                        },
                        complete: function() {
                            $('#test-connection').prop('disabled', false).text('Testa anslutning');
                        }
                    });
                } else {
                    var errorMsg = (pingResponse && pingResponse.data) ? pingResponse.data : 'Kunde inte nå WordPress API';
                    showConnectionError('WordPress API-fel: ' + errorMsg);
                    // Återställ knapp
                    $('#test-connection').prop('disabled', false).text('Testa anslutning');
                }
            },
            error: function(xhr, status, error) {
                var errorMsg = getDetailedErrorMessage(xhr, status, error);
                showConnectionError('WordPress-anslutningsfel: ' + errorMsg);
                // Återställ knapp
                $('#test-connection').prop('disabled', false).text('Testa anslutning');
            }
        });
    }
    
    /**
     * Visar ett anslutningsfel på ett användarvänligt sätt
     */
    function showConnectionError(message) {
        // Säkerställ att message är en sträng
        if (typeof message === 'object') {
            // Försök extrahera ett meningsfullt felmeddelande från objektet
            if (message.message) {
                message = message.message;
            } else if (message.error) {
                message = message.error;
            } else if (message.data && typeof message.data === 'object' && message.data.message) {
                message = message.data.message;
            } else if (message.responseJSON && message.responseJSON.message) {
                message = message.responseJSON.message;
            } else {
                // Sista utväg: konvertera hela objektet till JSON
                try {
                    message = JSON.stringify(message);
                } catch (e) {
                    message = 'Okänt fel (kunde inte formatera felmeddelande)';
                }
            }
        }
        
        $('#connection-status').html('<span class="error">Anslutningsfel: ' + message + '</span>');
        $('#test-results').html('<div class="notice notice-error"><p>Anslutningstest misslyckades: ' + message + '</p></div>').show();
        
        // Återställ testknappen
        $('#test-connection').prop('disabled', false).text('Testa anslutning');
        
        // Logga felet för felsökning
        console.error('Anslutningsfel:', message);
    }
    
    /**
     * Generera detaljerade felmeddelanden baserat på HTTP-fel
     */
    function getDetailedErrorMessage(xhr, status, error) {
        var detailedError = error || 'Okänt fel';
        
        if (status === 'timeout') {
            detailedError = 'Tidsgräns uppnådd vid anslutning';
        } else if (xhr && xhr.status === 0) {
            detailedError = 'Kunde inte ansluta. Kontrollera din internetanslutning eller om CORS-problem föreligger.';
        } else if (xhr && xhr.status === 403) {
            detailedError = 'Tillträde nekad (403). Sessionen kan ha gått ut eller så saknas behörighet.';
        } else if (xhr && xhr.status === 404) {
            detailedError = 'API-endpunkten kunde inte hittas (404).';
        } else if (xhr && xhr.status >= 500) {
            detailedError = 'Serverfel (' + xhr.status + '). Vänligen försök igen senare.';
        }
        
        console.log('Detaljerad felinformation:', {
            error: detailedError,
            status: status,
            xhr_status: xhr ? xhr.status : 'N/A',
            responseText: xhr ? xhr.responseText : 'Inget svar'
        });
        
        return detailedError;
    }

    /**
     * Visa formulärets sparningsstatus
     */
    function formSaved(success, message) {
        if (success) {
            // Visa ett snabbmeddelande som försvinner efter några sekunder
            var $notification = $('<div class="notice notice-success is-dismissible form-save-success"><p><strong>Inställningarna sparades!</strong></p></div>');
            $('#bokforehandel-settings-form').before($notification);
            
            // Sätt en timeout för att ta bort meddelandet
            setTimeout(function() {
                $notification.fadeOut(500, function() {
                    $(this).remove();
                });
            }, 3000);
        } else {
            // Visa felmeddelande
            var errorMessage = message || 'Ett okänt fel uppstod vid sparning av inställningar.';
            var $notification = $('<div class="notice notice-error is-dismissible"><p><strong>Fel vid sparning:</strong> ' + errorMessage + '</p></div>');
            $('#bokforehandel-settings-form').before($notification);
        }
    }

    function saveSettings(form) {
        // Kolla om det finns obligatoriska fält som saknas
        if (!validateRequired(form)) {
            showNotice('Fyll i alla obligatoriska fält.', 'error');
            return;
        }
        
        // Ta bort eventuella tidigare felmeddelanden
        $('.bokforehandel-required').removeClass('bokforehandel-required');
        
        // Hämta och spara aktiv flik på ett säkert sätt
        var activeTab = $('.bokforehandel-tabs-nav li a.ui-state-active').attr('href') || '#company';
        console.log('Aktiv flik:', activeTab);

        // Uppdatera knappen för att visa att vi processar
        var $button = form.find('button[type="submit"]');
        $button.text('Sparar...').prop('disabled', true);
        
        // Samla all formulärdata
        var formData = {};
        form.serializeArray().forEach(function(item) {
            formData[item.name] = item.value;
        });
        
        // Rensa potentiella fält som vi vill sätta manuellt
        delete formData.active_tab;
        delete formData.sync_enabled;
        
        // Samla bokföringskonton
        var accountingAccounts = {};
        $('input[name^="accounts["]').each(function() {
            var accountName = $(this).attr('name').match(/\[(.*?)\]/)[1];
            accountingAccounts[accountName] = $(this).val();
        });
        
        // Samla betalningskonton
        var paymentAccounts = {};
        $('select[name^="payment_accounts["]').each(function() {
            var gateway = $(this).attr('name').match(/\[(.*?)\]/)[1];
            paymentAccounts[gateway] = $(this).val();
        });
        
        // Skapa AJAX-data objekt
        var ajaxData = {
            action: 'bokforehandel_save_settings',
            nonce: apiParams.nonce,
            active_tab: activeTab.replace('#', ''),
            accounting_accounts: JSON.stringify(accountingAccounts),
            payment_accounts: JSON.stringify(paymentAccounts)
        };
        
        // Lägg till sync_enabled endast om elementet finns, annars lämnas det helt ut
        if ($('#auto-sync').length > 0) {
            ajaxData.sync_enabled = $('#auto-sync').is(':checked') ? 'yes' : 'no';
            console.log('Auto-sync värde från #auto-sync:', ajaxData.sync_enabled);
        } else if ($('input[name="sync_enabled"]').length > 0) {
            var value = $('input[name="sync_enabled"]:checked').val();
            if (value) {
                ajaxData.sync_enabled = value;
                console.log('sync_enabled värde från input:', ajaxData.sync_enabled);
            } else {
                console.log('Inget markerat värde för sync_enabled hittades.');
            }
        } else {
            console.log('Inget sync_enabled element hittades.');
        }
        
        // Slå ihop med övrig formulärdata, så att våra manuellt satta värden inte skrivs över
        ajaxData = $.extend({}, formData, ajaxData);
        
        // Logga vad vi skickar för felsökning
        console.log('Spara inställningar AJAX-data:', ajaxData);
        
        // Skicka AJAX-anropet
        $.ajax({
            url: apiParams.ajax_url,
            type: 'POST',
            data: ajaxData,
            success: function(response) {
                $button.text('Spara inställningar').prop('disabled', false);
                
                if (response.success) {
                    showNotice('Inställningarna har sparats.', 'success');
                    
                    // Återställ aktiv flik
                    setTimeout(function() {
                        console.log('Återställer till flik:', activeTab);
                        // Använd setTimeout med längre väntetid för att säkerställa att DOM har uppdaterats först
                        setTimeout(function() {
                            // Kontrollera om vi använder jQuery UI tabs eller anpassad tab-lösning
                            var $tabs = $('#bokforehandel-tabs');
                            if ($tabs.hasClass('ui-tabs')) {
                                // jQuery UI tabs - använd tabs API
                                var tabIndex = $('.bokforehandel-tabs-nav li a[href="' + activeTab + '"]').parent().index();
                                $tabs.tabs('option', 'active', tabIndex);
                            } else {
                                // Anpassad tab-lösning
                                $('.bokforehandel-tabs-nav li a').removeClass('ui-state-active');
                                $('.bokforehandel-tabs-nav li a[href="' + activeTab + '"]').addClass('ui-state-active');
                                
                                // Dölj alla flikar först och visa sedan den valda fliken
                                $('.bokforehandel-tab').hide();
                                $(activeTab).show();
                            }
                            
                            console.log('Aktiv flik återställd till:', activeTab);
                        }, 200); // Öka tidsfördröjningen för bättre stabilitet
                    }, 100);
                } else {
                    showNotice('Fel: ' + (response.data || 'Okänt fel'), 'error');
                }
            },
            error: function(xhr, status, error) {
                $button.text('Spara inställningar').prop('disabled', false);
                formSaved(false, error);
                var $modal = $('<div class="bokforehandel-modal"><div class="bokforehandel-modal-content error"><span class="bokforehandel-modal-close">&times;</span><p>AJAX-fel: ' + error + '</p></div></div>');
                $('body').append($modal);
                $modal.show();
                
                $modal.find('.bokforehandel-modal-close').on('click', function() {
                    $modal.fadeOut(300, function() {
                        $modal.remove();
                    });
                });
                
                // Återställ aktiv flik även vid fel
                setTimeout(function() {
                    if ($('#bokforehandel-tabs').hasClass('ui-tabs')) {
                        $('#bokforehandel-tabs').tabs('option', 'active', $('.bokforehandel-tabs-nav li a.ui-state-active').parent().index());
                    } else {
                        $('.bokforehandel-tabs-nav li a').removeClass('ui-state-active');
                        $('.bokforehandel-tabs-nav li a[href="' + activeTab + '"]').addClass('ui-state-active');
                        $('.bokforehandel-tab').hide();
                        $(activeTab).show();
                    }
                }, 200);
            }
        });
    }

    // Automatiskt spara autosynkronisering när man togglar switchen
    $('#auto-sync-top, #auto-sync').on('change', function() {
        var isChecked = $(this).is(':checked');
        
        // Synkronisera båda kontrollerna
        $('#auto-sync-top, #auto-sync').prop('checked', isChecked);
        
        // Uppdatera status-text
        $('.status-text').text(isChecked ? 'Aktiverad' : 'Inaktiverad');
        
        // Visa laddningsindikator
        $('.status-text').prepend('<span class="saving-indicator" style="margin-right: 5px; opacity: 0.6;">(Sparar...)</span>');
        
        // AJAX-anrop för att spara inställningen
        $.ajax({
            url: apiParams.ajax_url,
            type: 'POST',
            data: {
                action: 'bokforehandel_update_auto_sync',
                nonce: apiParams.nonce,
                sync_enabled: isChecked ? 'yes' : 'no'
            },
            success: function(response) {
                // Ta bort laddningsindikator
                $('.saving-indicator').remove();
                
                if (response.success) {
                    // Visa en diskret notis som försvinner efter kort tid
                    var noticeText = isChecked ? 'Autosynkronisering har aktiverats!' : 'Autosynkronisering har inaktiverats!';
                    showNotice(noticeText, 'success', 2000);
                } else {
                    // Visa felmeddelande
                    showNotice('Fel: ' + (response.data || 'Kunde inte spara inställningen'), 'error');
                }
            },
            error: function(xhr, status, error) {
                // Ta bort laddningsindikator
                $('.saving-indicator').remove();
                
                // Visa AJAX-fel
                showNotice('AJAX-fel: ' + error, 'error');
            }
        });
    });

    // Funktion för att skapa en stängbar notis
    function createDismissibleNotice(message, type) {
        var $notice = $('<div class="notice notice-' + type + ' is-dismissible"><p>' + message + '</p><button type="button" class="notice-dismiss"><span class="screen-reader-text">Stäng meddelande</span></button></div>');
        
        // Lägg till klickhändelse för stängning
        $notice.find('.notice-dismiss').on('click', function() {
            $notice.fadeOut(300, function() {
                $(this).remove();
            });
        });
        
        return $notice;
    }

    // Visa meddelande med stängfunktion
    function showNotice(message, type) {
        type = type || 'success';
        
        // Ta bort befintliga notiser av samma typ
        $('.notice.notice-' + type).remove();
        
        // Skapa och infoga notis
        var $notice = createDismissibleNotice(message, type);
        $('.wrap h1').after($notice);
        
        // Auto-stäng framgångsnotiser efter 5 sekunder
        if (type === 'success') {
            setTimeout(function() {
                $notice.fadeOut(300, function() {
                    $(this).remove();
                });
            }, 5000);
        }
    }

    /**
     * Returnerar en läsbar namn på en operation
     */
    function getOperationName(operation) {
        switch(operation) {
            case 'all': return 'Allt';
            case 'woo': return 'WooCommerce-ordrar';
            case 'klarna': return 'Klarna-utbetalningar';
            case 'stripe': return 'Stripe-utbetalningar';
            case 'order': return 'WooCommerce-order';
            default: return operation || 'Okänd';
        }
    }

})(jQuery);